<?php

namespace App\Http\Controllers\Registration;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Validation\ValidationException;

use GuzzleHttp\Client;
use Illuminate\Support\Facades\Storage;
use Log;
use DB;
use Exception;
use Carbon\Carbon;
use Maatwebsite\Excel\Facades\Excel;

use App\Models\Registration\Registration;
use App\Models\Tools\GasToolUsages;
use App\Models\Tools\FuelUsages;
use App\Models\Tools\GasTools;
use App\Models\Tools\Fuels;
use App\Exports\RegisterExport;
use App\Exports\RegisterReportExport;
use App\Http\Controllers\RegistrationCapelController;
use App\Models\KelengkapanData\KelengkapanData;
use App\Models\Geo\Area;
use App\Models\Geo\AreaInfo;
use App\Helpers\GaClient;

class RegistrationController extends Controller
{

    const DEFAULT_PER_PAGE = 10;

    public function __construct()
    {
        
    }

    private function buildSearch($data, Request $request)
    {
        $data = $data->where('nama', 'like', '%'.$request->search.'%')
                ->orWhere('ktp', 'like', '%'.$request->search.'%')
                ->orWhere('no_telpon', 'like', '%'.$request->search.'%')
                ->orWhere('email', 'like', '%'.$request->search.'%');

        return $data;
    }
    
    // REGISTER
    public function index(Request $request)
    {
        try {
            validation($request, [
                'page' => 'required|numeric',
                'perPage' => 'required|numeric',
                'search' => regexValidString()
            ]);
        } catch (ValidationException $e) {
            Log::error('RegistrationController - index - validasi : '. json_encode($e->validator->errors()));
            return response()->json([
                'status' => false,
                'message' => 'Failed validation',
                'errors' => $e->validator->errors()
            ], 422);
        }

        try {

            $perPage = self::DEFAULT_PER_PAGE;
            
            $registrations = Registration::query();
            
            if ($request->has('perPage')) {
                $perPage = $request->perPage;
            }
            if ($request->has('search')) {
                $registrations = $this->buildSearch($registrations, $request);
            }

            $registrations = $registrations->orderBy('created_at', 'DESC');
            $registrations = $registrations->paginate($perPage);
            
            return response()->json([
                'status' => true,
                'message' => 'Berhasil mengambil data',
                'data' => $registrations
            ], 200);

        } catch (Exception $e) {
            Log::error('RegistrationController - index : '. $e->getMessage());
            return response()->json([
                'status' => false,
                'message' => 'Gagal menghubungkan ke server'
            ], 500);
        }
    }

    public function detail(Request $request, $id)
    {
        try {
            $request->merge([
                'id' => $id
            ]);

            validation($request, [
                'id' => 'required|numeric|exists:registrations,id'
            ]);
        } catch (ValidationException $e) {
            Log::error('RegistrationController - detail - validasi : '. json_encode($e->validator->errors()));
            return response()->json([
                'status' => false,
                'message' => 'Failed validation',
                'errors' => $e->validator->errors()
            ], 422);
        }

        try {

            $registrations = Registration::with([
                'province',
                'town',
                'district',
                'village',
                'housingArea',
                'buildingType',
                'statusLocation',
                'electricalPower',
                'fuelUsages', 
                'gasToolUsages'
            ])
            ->find($id);

            $gasTools = GasTools::where('is_deleted', 0)
            ->orderBy('sort', 'ASC')
            ->get();

            $fuels = Fuels::where('is_deleted', 0)
            ->orderBy('sort', 'ASC')
            ->get();
            
            return response()->json([
                'status' => true,
                'message' => 'Berhasil mengambil data',
                'data' => [
                    'registrations' => $registrations,
                    'gas_tools' => $gasTools,
                    'fuels' => $fuels
                ]
            ], 200);

        } catch (Exception $e) {
            Log::error('RegistrationController - index : '. $e->getMessage());
            return response()->json([
                'status' => false,
                'message' => 'Gagal menghubungkan ke server'
            ], 500);
        }
    }

    public function exportRegister(Request $request)
    {
        return Excel::download(new RegisterExport($request, 'register-list'), 'register-data.xlsx');
    }
    // END REGISTER

    // REGISTER LOG
    public function indexLog(Request $request)
    {
        try {
            validation($request, [
                'page' => 'required|numeric',
                'perPage' => 'required|numeric',
                'search' => regexValidString()
            ]);
        } catch (ValidationException $e) {
            Log::error('RegistrationController - index - validasi : '. json_encode($e->validator->errors()));
            return response()->json([
                'status' => false,
                'message' => 'Failed validation',
                'errors' => $e->validator->errors()
            ], 422);
        }

        try {

            $perPage = self::DEFAULT_PER_PAGE;
            
            $registrations = Registration::query();
            
            if ($request->has('perPage')) {
                $perPage = $request->perPage;
            }
            if ($request->has('search')) {
                $registrations = $this->buildSearch($registrations, $request);
            }

            $registrations = $registrations->orderBy('created_at', 'DESC');
            $registrations = $registrations->paginate($perPage);
            
            return response()->json([
                'status' => true,
                'message' => 'Berhasil mengambil data',
                'data' => $registrations
            ], 200);

        } catch (Exception $e) {
            Log::error('RegistrationController - index : '. $e->getMessage());
            return response()->json([
                'status' => false,
                'message' => 'Gagal menghubungkan ke server'
            ], 500);
        }
    }

    public function exportRegisterLog(Request $request)
    {
        return Excel::download(new RegisterExport($request, 'register-log'), 'register-log-data.xlsx');
    }

    public function syncRegisterLog(Request $request)
    {
        try {
            validation($request, [
                'id' => 'required|numeric|exists:registrations,id'
            ]);
        } catch (ValidationException $e) {
            Log::error('RegistrationController - syncRegisterLog - validasi : '. json_encode($e->validator->errors()));
            return response()->json([
                'status' => false,
                'message' => 'Failed validation',
                'errors' => $e->validator->errors()
            ], 422);
        }

        try {

            $Registration = Registration::find($request->id);

            if ($Registration->is_send === 1) {
                return response()->json([
                    'status' => false,
                    'message' => 'Data sudah tersinkronisasi'
                ], 400);    
            }

            $sync_data = (new RegistrationCapelController)->callApiRegsitrasi($Registration);

            if ($sync_data == 'failed') {
                return response()->json([
                    'status' => false,
                    'message' => 'Gagal sinkronisasi data'
                ], 400);
            } else {
                return response()->json([
                    'status' => true,
                    'message' => 'Berhasil sinkronisasi data'
                ], 200);
            }
        } catch (Exception $e) {
            Log::error('RegistrationController - syncRegisterLog : '. $e->getMessage());
            return response()->json([
                'status' => false,
                'message' => 'Gagal menghubungkan ke server'
            ], 500);
        }
    }
    // END REGISTER LOG

    // REGISTER REPORT

    private function buildSearchReport($data, Request $request)
    {
        if ($request->search != "000") {
            $data = $data->whereHas('town', function ($q) use($request)  {
                $q->whereHas('areaInfo', function ($q2) use($request) {
                    $q2->where('area_id', $request->search)
                    ->where('is_deleted', 0);
                });
            });
        }

        return $data;
    }

    public function indexReport(Request $request)
    {
        try {
            validation($request, [
                'page' => 'required|numeric',
                'perPage' => 'required|numeric',
                'search' => regexValidString(),
                'date' => 'date'
            ]);
        } catch (ValidationException $e) {
            Log::error('RegistrationController - index - validasi : '. json_encode($e->validator->errors()));
            return response()->json([
                'status' => false,
                'message' => 'Failed validation',
                'errors' => $e->validator->errors()
            ], 422);
        }

        try {

            $perPage = self::DEFAULT_PER_PAGE;
            
            $registrations = Registration::query();

            if ($request->has('date')) {
                $temp_date = Carbon::parse($request->date)->format('Y-m-d');
                $registrations = $registrations->whereDate('created_at', $temp_date);
            }
            
            if ($request->has('perPage')) {
                $perPage = $request->perPage;
            }
            if ($request->has('search')) {
                $registrations = $this->buildSearchReport($registrations, $request);
            }

            $registrations = $registrations->orderBy('created_at', 'DESC');
            $registrations = $registrations->paginate($perPage);
            
            return response()->json([
                'status' => true,
                'message' => 'Berhasil mengambil data',
                'data' => $registrations
            ], 200);

        } catch (Exception $e) {
            Log::error('RegistrationController - index : '. $e->getMessage());
            return response()->json([
                'status' => false,
                'message' => 'Gagal menghubungkan ke server'
            ], 500);
        }
    }

    public function exportRegisterReport(Request $request)
    {
        return Excel::download(new RegisterReportExport($request), 'register-report.xlsx');
    }

    public function sendEmailRegisterReport(Request $request, GaClient $GaClient)
    {
        try {
            validation($request, [
                'area_id' => 'required|numeric|exists:areas,id,is_deleted,0',
                'date' => 'date'
            ]);
        } catch (ValidationException $e) {
            Log::error('RegistrationController - sendEmailRegisterReport - validasi : '. json_encode($e->validator->errors()));
            return response()->json([
                'status' => false,
                'message' => 'Failed validation',
                'errors' => $e->validator->errors()
            ], 422);
        }

        try {

            $Area = Area::where('id', $request->area_id)
            ->where('is_deleted', 0)
            ->first();

            if (empty($Area)) {
                return response()->json([
                    'status' => false,
                    'message' => 'Data Area tidak ditemukan',
                    'data' => []
                ], 400);
            }

            if (empty($Area->manager_email)) {
                return response()->json([
                    'status' => false,
                    'message' => 'Data Email Manager Area tidak ditemukan',
                    'data' => []
                ], 400);
            }

            $temp_file_path = 'public/file_upload/register-report'.Carbon::now()->format('Ymd-His').'.xlsx';
            $excelPath = Excel::store(new RegisterReportExport($request), $temp_file_path);
            
            // SEND EMAIL
            $to = $Area->manager_email;
            $subject = 'Rekapitulasi data calon pelanggan';
            $body = '';
            $attachment = [
                'file_path' => storage_path('app/' . $temp_file_path),
                'file_name' => 'register-report-'.Carbon::now()->format('Ymd').'.xlsx'
            ];
            

            $year = date("Y");

            $data = [
                'date_start' => Carbon::parse($request->date)->format('d/m/Y'),
                'area_name' => $Area->name
            ];
            $html = view('mail.report_registrasi_email', [
                'data' => $data,
                'year' => $year,
            ])->render();

            $body = $html;

            $response_ga = $GaClient->sendEmail($to, $subject, $body, [$attachment]);

            if ($response_ga) {
                unlink(storage_path('app/' . $temp_file_path));

                return response()->json([
                    'status' => true,
                    'message' => 'Berhasil kirim email',
                    'data' => []
                ], 200);    
            }
            // END SEND EMAIL

        } catch (Exception $e) {
            Log::error('RegistrationController - sendEmailRegisterReport - error : '. $e->getMessage());
            return response()->json([
                'status' => false,
                'message' => 'Gagal menghubungkan ke server',
                'data' => []
            ], 500);
        }
    }

    public function getArea(Request $request)
    {
        try {
            
            $area = Area::where('is_deleted', 0)->get();

            return response()->json([
                'status' => true,
                'message' => 'Berhasil mengambil data',
                'data' => $area
            ], 200);

        } catch (Exception $e) {
            Log::error('RegistrationController - index : '. $e->getMessage());
            return response()->json([
                'status' => false,
                'message' => 'Gagal menghubungkan ke server'
            ], 500);
        }
    }
    // END REGISTER REPORT
}
