<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Validation\ValidationException;

use App\Models\Registration;
use App\Models\Tools\GasToolUsages;
use App\Models\Tools\FuelUsages;
use GuzzleHttp\Client;
use Illuminate\Support\Facades\Storage;
use Log;
use DB;
use Exception;
use Carbon\Carbon;
use App\Jobs\RegistrationRelyonJob;
use GuzzleHttp\Exception\RequestException;
use App\Services\MinioService;

class RegistrationCapelController extends Controller
{

    protected $headers;
    protected $minioService;

    public function __construct()
    {
        $this->headers = [
            'client_id' => env('RELYON_CLIENT_ID', ''),
            'client_secret' => env('RELYON_CLIENT_SECRET', ''),
            'Authorization' => 'Bearer '.env('RELYON_TOKEN', '')
        ];

        $this->minioService = new MinioService;
    }

    public function postRegister(Request $request)
    {
        
        try {
            validation($request, [
                'kebutuhan_pelanggan' => 'required|numeric|in:1,2',
                'lokasi_pemasangan_provinsi' => 'required|numeric|exists:geo_provinces,id,is_deleted,0',
                'lokasi_pemasangan_kota' => 'required|numeric|exists:geo_towns,id,is_deleted,0',
                'lokasi_pemasangan_kecamatan' => 'required|numeric|exists:geo_districts,id,is_deleted,0',
                'lokasi_pemasangan_kelurahan' => 'required|numeric|exists:geo_villages,id,is_deleted,0',
                'lokasi_pemasangan_perumahan_jalan_utama' => 'numeric|exists:housing_areas,id,is_deleted,0',
                'lokasi_pemasangan_nama_perumahan_jalan_utama' => regexValidString(),
                'lokasi_pemasangan_alamat' => 'required|'.regexValidString(),
                'lokasi_pemasangan_rt' => regexValidNumber().'|max:3',
                'lokasi_pemasangan_rw' => regexValidNumber().'|max:3',
                'lokasi_pemasangan_kode_pos' => regexValidNumber().'|max:5',
                'lokasi_pemasangan_jenis_bangunan' => 'required|numeric|exists:building_types,id,is_deleted,0',
                'lokasi_pemasangan_status_kepemilikan' => 'required|numeric|exists:status_locations,id,is_deleted,0',
                'lokasi_pemasangan_nama_pemilik_bangunan' => 'required_if:lokasi_pemasangan_status_kepemilikan,2|'.regexValidString(),
                'lokasi_pemasangan_no_telpon' => 'required_if:lokasi_pemasangan_status_kepemilikan,2|'.regexValidPhone().'|min:11|max:16',
                'lokasi_pemasangan_status_lokasi_pemasangan' => 'required|'.regexValidNumber().'|in:0,1',
                'lokasi_pemasangan_latitude' => 'required|numeric',
                'lokasi_pemasangan_longitude' => 'required|numeric',
    
                'penggunaan_nama' => 'required|'.regexValidString(),
                'penggunaan_ktp' => 'required|'.regexValidNumber().'|min:14|max:16',
                'penggunaan_no_telpon' => 'required|'.regexValidPhone().'|min:11|max:16',
                'penggunaan_email' => 'required|email',
                'penggunaan_energi_listrik' => 'required|numeric|exists:electrical_powers,id,is_deleted,0',
                'penggunaan_bahan_bakar' => 'required|array',
                'penggunaan_bahan_bakar.*.id' => 'numeric|exists:fuels,id,is_deleted,0',
                'penggunaan_bahan_bakar.*.value' => 'numeric|min:0|max:10',
                'penggunaan_peralatan' => 'required_if:kebutuhan_pelanggan,2|array',
                'penggunaan_peralatan.*.id' => 'numeric|exists:fuels,id,is_deleted,0',
                'penggunaan_peralatan.*.value' => 'numeric|min:0|max:10',
                'penggunaan_ktp_file' => 'required|image|mimes:jpeg,png,jpg|max:5000',
                'penggunaan_rekening_listrik_file' => 'required_if:penggunaan_energi_listrik,<3|image|mimes:jpeg,png,jpg|max:5000',
            ]);

        } catch (ValidationException $e) {
            return response()->json([
                'status' => false,
                'message' => 'Failed validation',
                'errors' => $e->validator->errors()
            ], 422);
        }

        DB::beginTransaction();
        try {
            // CEK KTP
            // $data = Registration::where('ktp', $request->penggunaan_ktp)
            // ->orderBy('created_at', 'DESC')
            // ->first();

            // if (!empty($data)) {
            //     return response()->json([
            //         'status' => false,
            //         'message' => 'duplicate data',
            //         'errors' => ''
            //     ], 422);
            // }
            // END CEK KTP

            // SAVE FILE TEMPORARY
            // if (!Storage::cloud()->exists('file_upload')) {
            //     Storage::cloud()->makeDirectory('file_upload');
            // }

            
            
            $path_ktp = "";
            if ($request->has('penggunaan_ktp_file')) {

                $file = $request->file('penggunaan_ktp_file');
                $minioFilePath = time()."_".$file->getClientOriginalName(); 

                $this->minioService->uploadFile($file, $minioFilePath);

                $path_ktp = $minioFilePath;
            }

            $path_rekening_listrik = "";
            if ($request->has('penggunaan_rekening_listrik_file')) {
                
                $file = $request->file('penggunaan_rekening_listrik_file');
                $minioFilePath = time()."_".$file->getClientOriginalName(); 

                $this->minioService->uploadFile($file, $minioFilePath);

                $path_rekening_listrik = $minioFilePath;
            }
            // END SAVE FILE TEMPORARY
            
            $Registration = Registration::create([
                'kebutuhan_pelanggan' => $request->kebutuhan_pelanggan,
                'geo_province_id' => $request->lokasi_pemasangan_provinsi,
                'geo_town_id' => $request->lokasi_pemasangan_kota,
                'geo_district_id' => $request->lokasi_pemasangan_kecamatan,
                'geo_village_id' => $request->lokasi_pemasangan_kelurahan,
                'housing_area_id' => $request->has('lokasi_pemasangan_perumahan_jalan_utama') ? $request->lokasi_pemasangan_perumahan_jalan_utama : null,
                'nama_perumahan_jalan_utama' => $request->has('lokasi_pemasangan_nama_perumahan_jalan_utama') ? $request->lokasi_pemasangan_nama_perumahan_jalan_utama : "",
                'alamat' => $request->lokasi_pemasangan_alamat,
                'rt' => $request->has('lokasi_pemasangan_rt') ? $request->lokasi_pemasangan_rt : "",
                'rw' => $request->has('lokasi_pemasangan_rw') ? $request->lokasi_pemasangan_rw : "",
                'kode_pos' => $request->has('lokasi_pemasangan_kode_pos') ? $request->lokasi_pemasangan_kode_pos : "",
                'building_type_id' => $request->lokasi_pemasangan_jenis_bangunan,
                'nama_pemilik_bangunan' => $request->has('lokasi_pemasangan_nama_pemilik_bangunan') ? $request->lokasi_pemasangan_nama_pemilik_bangunan : "",
                'no_telpon_pemilik_bangunan' => $request->has('lokasi_pemasangan_no_telpon') ? $request->lokasi_pemasangan_no_telpon : "",
                'status_lokasi_pemasangan' => $request->lokasi_pemasangan_status_lokasi_pemasangan,
                'latitude' => $request->lokasi_pemasangan_latitude,
                'longitude' => $request->lokasi_pemasangan_longitude,
                'status_location_id' => $request->lokasi_pemasangan_status_kepemilikan,

                'nama' => $request->penggunaan_nama,
                'ktp' => $request->penggunaan_ktp,
                'no_telpon' => $request->penggunaan_no_telpon,
                'email' => $request->penggunaan_email,
                'electrical_power_id' => $request->penggunaan_energi_listrik,
                'ktp_url' => $path_ktp,
                'rekening_listrik_url' => $path_rekening_listrik,
                'is_send' => 0,
            ]);

            foreach ($request->penggunaan_bahan_bakar as $key => $val) {
                $FuelUsages = FuelUsages::create([
                    'registration_id' => $Registration->id,
                    'fuel_id' => $val['id'],
                    'value_unit' => $val['value'],
                ]);
            }
            if ($request->has('penggunaan_peralatan')) {
                foreach ($request->penggunaan_peralatan as $key => $val) {
                    $GasToolUsages = GasToolUsages::create([
                        'registration_id' => $Registration->id,
                        'gas_tool_id' => $val['id'],
                        'amount' => $val['value'],
                    ]);
                }
            }

            RegistrationRelyonJob::dispatch($Registration);
            
            DB::commit();

            // CEK KATALOG PRODUK
            $data_katalog_produk = [];
            $katalog_produk = $this->checkKatalogProduk($request);
            
            if (gettype($katalog_produk) == "object" && $katalog_produk->status) {
                $data_katalog_produk = $katalog_produk->data;
            }
            // END CEK KATALOG PRODUK

            return response()->json([
                "status" => true,
                "data" => [
                    'registrasi' => $Registration,
                    'katalog_produk' => $data_katalog_produk
                ],
                "message" => "Berhasil menyimpan pendaftaran"
            ]);
            
        } catch (Exception $e) {
            DB::rollBack();
            Log::error('postRegister : '. $e->getMessage());
            return response()->json([
                'status' => false,
                'message' => 'Gagal menghubungkan ke server'
            ], 500);
        }
    }

    public function uploadFile($url, $file, $originalName, $originalExtension)
    {
        try {

            $client = new Client(['verify' => false]);
            
            $request = [
                'multipart' => [
                    [
                        'name' => 'file',
                        // 'contents' => fopen($file, 'r'),
                        'contents' => $file,
                        // 'filename' => $file->getClientOriginalName(),
                        // 'ext' => $file->getClientOriginalExtension()
                        'filename' => $originalName,
                        'ext' => $originalExtension
                    ]
                ],
                'headers' => [
                    'client_id' => env('RELYON_CLIENT_ID', ''),
                    'client_secret' => env('RELYON_CLIENT_SECRET', ''),
                    'Authorization' => 'Bearer '.env('RELYON_TOKEN', '')
                ]
            ];

            $response = $client->request('POST', $url, $request);
            
            $result = json_decode((string)$response->getBody());
            if ($response->getStatusCode() == 200 && $result->status == true) {
                return $result->path;
            }
            
            return "";

        } catch (\Exception $e) {
            Log::info('RegistrationCapelController - uploadFile error : '.$e->getMessage());
            return "";
        }
    }

    public static function callApiRegsitrasi($data)
    {
        DB::beginTransaction();
        try {

            $client = new Client(['verify' => false]);
            $url = env('RELYON_REGISTRATION', '');

            $temp_fuel_usages = [];
            foreach ($data->fuelUsages as $key => $val) {
                array_push($temp_fuel_usages, [
                    "fuel_id" => $val->fuel_id,
                    "amount" => $val->value_unit
                ]);
            }

            $gas_tool_usages = [];
            if (!empty($data->gasToolUsages)) {
                foreach ($data->gasToolUsages as $key => $val) {
                    array_push($gas_tool_usages, [
                        "gas_tool_id" => $val->gas_tool_id,
                        "amount" => $val->amount
                    ]);
                }
            }

            $folder = storage_path('app/public/');
            $ktp_url = $folder . $data->ktp_url;
            $rekening_listrik_url = $folder . $data->rekening_listrik_url;
            
            $idcard_url = "";
            $electricity_bill_url = "";
            
            if ($data->ktp_url != null && $data->ktp_url != "") {
                if ($data->ktp_url_relyon == null || $data->ktp_url_relyon == "") {
                    if (Storage::exists('public/'.$data->ktp_url)) {
                        $originalName = pathinfo(Storage::get($ktp_url), PATHINFO_FILENAME);
                        $originalExtension = pathinfo(Storage::get($ktp_url), PATHINFO_EXTENSION);
                        
                        $idcard_url = (new RegistrationCapelController)->uploadFile(env('RELYON_UPLOAD_ID_CARD'), fopen($ktp_url, 'r'), $originalName, $originalExtension);
        
                        if ($idcard_url != "") {
                            $data->ktp_url_relyon = $idcard_url;
                            $data->save();
                            DB::commit();
                        }
                    }
                } else {
                    $idcard_url = $data->ktp_url_relyon;
                }
            }

            if ($data->rekening_listrik_url != null && $data->rekening_listrik_url != "") {
                if ($data->rekening_listrik_url_relyon == null || $data->rekening_listrik_url_relyon == "") {
                    if (Storage::exists('public/'.$data->rekening_listrik_url)) {
                        $originalName = pathinfo(Storage::get($rekening_listrik_url), PATHINFO_FILENAME);
                        $originalExtension = pathinfo(Storage::get($rekening_listrik_url), PATHINFO_EXTENSION);
                        
                        $electricity_bill_url = (new RegistrationCapelController)->uploadFile(env('RELYON_UPLOAD_EBILL'), fopen($rekening_listrik_url, 'r'), $originalName, $originalExtension);
        
                        if ($electricity_bill_url != "") {
                            $data->rekening_listrik_url_relyon = $electricity_bill_url;
                            $data->save();
                            DB::commit();
                        }
                    }
                } else {
                    $electricity_bill_url = $data->rekening_listrik_url_relyon;
                }
            }

            if ($idcard_url != "") {
                if (
                    ($data->rekening_listrik_url != null && $data->rekening_listrik_url != "") && 
                    $electricity_bill_url == ""    
                ) {
                    Log::error("callApiRegsitrasi error : gagal upload rekening_listrik = " . json_encode([$data->rekening_listrik_url, $electricity_bill_url]));
                    return "Failed";
                }
                $request = [
                    'headers' => [
                        'client_id' => env('RELYON_CLIENT_ID', ''),
                        'client_secret' => env('RELYON_CLIENT_SECRET', ''),
                        'Authorization' => 'Bearer '.env('RELYON_TOKEN', '')
                    ],
                    'json' => [
                        "communication_media_id" => "1",
                        "status_location_id" => $data->status_location_id,
                        "building_type_id" => $data->building_type_id,
                        "electrical_power_id" => $data->electrical_power_id,
                        "geo_province_id" => $data->geo_province_id,
                        "geo_town_id" => $data->geo_town_id,
                        "geo_district_id" => $data->geo_district_id, 
                        "geo_village_id" => $data->geo_village_id,
                        "housing_area_id" => $data->housing_area_id,
                        "housing_area_other" => $data->nama_perumahan_jalan_utama,
                        "idcard_number" => $data->ktp,
                        "name" => $data->nama,
                        "mobile_phone" => $data->no_telpon,
                        "address" => $data->alamat,
                        "rt" => $data->rt,
                        "rw" => $data->rw,
                        "email" => $data->email,
                        "postal_code" => $data->kode_pos,
                        "building_area" => "1",
                        "idcard_type_id" => "1",
                        "idcard_url" => $idcard_url,
                        "signature_url" => "",
                        "electricity_bill_url" => $electricity_bill_url,
                        "is_governance_property" => $data->status_lokasi_pemasangan,
                        "registered_by" => "0",
                        "registered_date" => Carbon::now()->timestamp,
                        "latitude" => $data->latitude, 
                        "longitude" => $data->longitude, 
                        "status_location_other" => "",
                        "building_type_other" => "",
                        "extension_zone_type_id" => "",
                        "fuel_usages"=> $temp_fuel_usages,
                        "gas_tool_usages" => $gas_tool_usages,
                        "customer_segmen" => $data->kebutuhan_pelanggan,
                        "client_name" => "PGN Register Web Platform",
                        "nama_pemilik_bangunan" => $data->nama_pemilik_bangunan,
                        "no_telpon_pemilik_bangunan" => $data->no_telpon_pemilik_bangunan
                    ]
                ];

                Log::info('RegistrationCapelController - callApiRegsitrasi - payload : '.json_encode($request));
    
                $response = $client->request('POST', $url, $request);
                
                $result = json_decode((string)$response->getBody());
                if ($response->getStatusCode() == 200 && $result->status == true) {
                    
                    $data->form_id = $result->form_id;
                    $data->is_send = 1;
                    $data->save();
    
                    Log::info('RegistrationCapelController - callApiRegsitrasi success for id : '.$data->id . ", form_id : " .$result->form_id);
                    return $result;
                } else {
                    
                    Log::error('RegistrationCapelController - callApiRegsitrasi failed for id : '.$data->id);
                }
            }

            DB::commit();
            return "failed";

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('RegistrationCapelController - callApiRegsitrasi error : '.$e->getMessage());
            return "failed";
        }
    }

    public function checkDuplicateData(Request $request)
    {
        try {
            validation($request, [
                'ktp' => regexValidNumber().'|size:16',
                'no_telpon' => regexValidPhone().'|min:11|max:16'
            ]);
        } catch (ValidationException $e) {
            Log::error('checkDuplicateData - validasi : '. json_encode($e->validator->errors()));
            return response()->json([
                'status' => false,
                'message' => 'Failed validation',
                'errors' => $e->validator->errors()
            ], 422);
        }

        // BYPASS
        // return response()->json([
        //     'status' => true,
        //     'message' => 'Berhasil mengambil data',
        //     'data' => [
        //         [
        //             'form_id' => '123',
        //             'address' => 'Jl. bawang no 12 karawaci tangerang banten'
        //         ],
        //         [
        //             'form_id' => '4321',
        //             'address' => 'Jl. bawang no 13 karawaci tangerang banten'
        //         ]
        //     ]
        // ], 200);
        // END BYPASS

        try {
            
            $client = new Client(['verify' => false]);
            $url = env('RELYON_CHECK_DUPLICATE_DATA', '');

            $payload = [
                'headers' => $this->headers,
                'json' => [
                    'idcard_type' => 1
                ]
            ];
            if ($request->has('ktp')) {
                $payload['json']['idcard_number'] = $request->ktp;
            }
            if ($request->has('no_telpon')) {
                $payload['json']['mobile_phone'] = $request->no_telpon;
            }
            
            Log::info('checkDuplicateData - callApiCheckDuplicateData payload : '.json_encode($payload));

            $response = $client->request('POST', $url, $payload);

            $result = json_decode((string)$response->getBody());
            if ($response->getStatusCode() == 200 && $result->status == true) {
                return $result;
            } else {
                return response()->json([
                    'status' => false,
                    'message' => 'Gagal mengambil data',
                    'data' => []
                ], 400);
                Log::error('checkDuplicateData - callApiCheckDuplicateData failed : '.json_encode($result));
            }

        } catch (Exception $e) {
            Log::error('checkDuplicateData : '. $e->getMessage());
            return response()->json([
                'status' => false,
                'message' => 'Gagal menghubungkan ke server'
            ], 500);
        }
    }

    public function checkKatalogProduk($data)
    {
        try {
            
            $client = new Client(['verify' => false]);
            $url = env('RELYON_CHECK_KATALOG_PRODUK', '');

            $payload = [
                'headers' => $this->headers,
                'json' => [
                    "housing_area_id" => $data->lokasi_pemasangan_perumahan_jalan_utama,
                    "building_type_id" => $data->lokasi_pemasangan_jenis_bangunan,
                    "electrical_power_id" => $data->penggunaan_energi_listrik,
                    "is_governance_property" => $data->lokasi_pemasangan_status_lokasi_pemasangan
                ]
            ];
            
            Log::info('checkKatalogProduk - callApiCheckKatalogProduk payload : '.json_encode($payload));

            $response = $client->request('POST', $url, $payload);

            $result = json_decode((string)$response->getBody());
            if ($response->getStatusCode() == 200 && $result->status == true) {
                return $result;
            } else {
                return [
                    'status' => false,
                    'message' => 'Gagal mengambil data',
                    'data' => []
                ];
                Log::error('checkKatalogProduk - callApiCheckKatalogProduk failed : '.json_encode($result));
            }

        } catch (Exception $e) {
            Log::error('checkKatalogProduk : '. $e->getMessage());
            return [
                'status' => false,
                'message' => 'Gagal menghubungkan ke server'
            ];
        }
    }
}
