// ** React Imports
import { useState, useEffect, Fragment } from 'react'
import { useRouter } from 'next/router'
import { useTheme } from '@mui/material/styles'

// ** MUI Imports
import Paper from '@mui/material/Paper'
import Table from '@mui/material/Table'
import TableRow from '@mui/material/TableRow'
import TableHead from '@mui/material/TableHead'
import TableBody from '@mui/material/TableBody'
import TableCell from '@mui/material/TableCell'
import TableContainer from '@mui/material/TableContainer'
import TablePagination from '@mui/material/TablePagination'
import Button from '@mui/material/Button'
import ButtonGroup from '@mui/material/ButtonGroup'
import CustomTextField from 'src/@core/components/mui/text-field'
import toast from 'react-hot-toast'
import CircularProgress from '@mui/material/CircularProgress'
import Fab from '@mui/material/Fab'
import Icon from 'src/@core/components/icon'
import Chip from '@mui/material/Chip'

// PGN
import { endPoint } from 'src/service/constans'
import { callGet, callExport, callPost } from 'src/service/api'

const columns = [
  { id: 'no', label: 'No', minWidth: 50 },
  {
    id: 'action',
    label: 'Action',
    minWidth: 100,
    align: 'center'
  },
  { id: 'is_send', label: 'Is Sync CRM', minWidth: 100 },
  { id: 'is_send_email', label: 'Is Send Email', minWidth: 100 },
  { id: 'nama_pelanggan', label: 'Nama Pelanggan', minWidth: 170 },
  { id: 'customer_id', label: 'No ID Pelanggan', minWidth: 170 },
  { id: 'town_name', label: 'Kota/Kabupaten', minWidth: 100 },
  { id: 'is_pk', label: 'Pelanggan Kecil', minWidth: 100 },
  { id: 'is_badan_hukum', label: 'Berbadan Hukum', minWidth: 100 },
  { id: 'is_npwp', label: 'Memiliki NPWP', minWidth: 100 },
  { id: 'npwp', label: 'NPWP Pelanggan', minWidth: 100 },
  { id: 'ktp', label: 'NIK Pelanggan', minWidth: 100 },
  { id: 'nama_ktp', label: 'Nama Sesuai NIK/NPWP', minWidth: 100 },
  { id: 'alamat_ktp', label: 'Alamat NIK/NPWP', minWidth: 100 },
  { id: 'no_telpon', label: 'HP', minWidth: 100 },
  { id: 'email', label: 'Email', minWidth: 100 }
]

const TableData = () => {
  const router = useRouter()
  const theme = useTheme()

  // ** States
  const [page, setPage] = useState(0)
  const [rowsPerPage, setRowsPerPage] = useState(10)
  const [search, setSearch] = useState('')
  const [isLoading, setIsLoading] = useState(false)
  const [responseList, setResponseList] = useState(null)
  const [dataList, setDataList] = useState([])
  const [loadingRows, setLoadingRows] = useState([])

  const handleChangePage = async (event, newPage) => {
    setPage(newPage)
    await getList(newPage, rowsPerPage)
  }

  const handleChangeRowsPerPage = async event => {
    setRowsPerPage(+event.target.value)
    setPage(0)
    getList(0, +event.target.value)
  }

  // PGN
  const [errors, setErrors] = useState({
    status: false,
    data: {},
    message: ''
  })

  useEffect(() => {
    if (errors.status) {
      if (errors.message == 'Failed validation' && errors.data.data.errors) {
        for (let field in errors.data.data.errors) {
          let errorMessages = errors.data.data.errors[field]
          errorMessages.forEach(errorMessage => {
            toast.error(errorMessage, {
              duration: 6000,
              style: {
                padding: '16px',
                color: theme.palette.error.main,
                border: '1px solid {theme.palette.error.main}'
              },
              iconTheme: {
                primary: theme.palette.error.main,
                secondary: theme.palette.error.contrastText
              }
            })
          })
        }
      } else {
        toast.error(errors.message, {
          duration: 6000,
          style: {
            padding: '16px',
            color: theme.palette.error.main,
            border: '1px solid {theme.palette.error.main}'
          },
          iconTheme: {
            primary: theme.palette.error.main,
            secondary: theme.palette.error.contrastText
          }
        })
      }

      setErrors({
        status: false,
        data: {},
        message: ''
      })
    }
  }, [errors])

  useEffect(() => {
    getList(page, rowsPerPage)
  }, [])

  const getList = async (temp_page, temp_rowsPerPage) => {
    setIsLoading(true)

    let params = {
      perPage: temp_rowsPerPage,
      page: temp_page + 1
    }
    if (search != '') {
      params.search = search
    }
    let response = await callGet(endPoint.getListKelengkapanDataLog, params)

    if (response.status) {
      setResponseList(response.data.data)
      setDataList(response.data.data.data)

      setIsLoading(false)
    } else {
      setResponseList(null)
      setDataList([])
      setErrors({
        status: true,
        data: response,
        message: response.data.message
          ? response.data.message
          : response.data.error
          ? response.data.error
          : 'Gagal menghubungkan ke server'
      })

      setIsLoading(false)
    }
  }

  const changeSearch = e => {
    setSearch(e.target.value)
  }

  const clickSearch = () => {
    getList(page, rowsPerPage)
  }

  const clickDetail = row => {
    router.push(`/register-detail/${row.id}`)
  }

  const clickExport = async () => {
    setIsLoading(true)

    let params = {}
    if (search != '') {
      params.search = search
    }
    let response = await callExport(endPoint.getExportKelengkapanDataLog, params)

    if (response.status) {
      console.log('response ', response)
      const url = window.URL.createObjectURL(new Blob([response.data]))

      const currentDate = new Date()
      const dateString = currentDate.toISOString().split('T')[0] // Ambil bagian tanggal dari timestamp ISO
      const fileName = `kelengkapan_data_log_${dateString}.xlsx`

      const link = document.createElement('a')
      link.href = url
      link.setAttribute('download', fileName)
      document.body.appendChild(link)
      link.click()

      window.URL.revokeObjectURL(url)

      setIsLoading(false)
    } else {
      setIsLoading(false)
      setErrors({
        status: true,
        data: response,
        message: response.data.message
          ? response.data.message
          : response.data.error
          ? response.data.error
          : 'Gagal menghubungkan ke server'
      })
    }
  }

  const syncData = async (row, rowIndex) => {
    setLoadingRows(prevLoadingRows => [...prevLoadingRows, rowIndex])

    let payload = {
      id: row.id
    }

    let response = await callPost(endPoint.postSyncKelengkapanDataLog, payload)

    if (response.status) {
      toast.success(response.message, {
        duration: 6000,
        style: {
          padding: '16px',
          color: theme.palette.success.main,
          border: '1px solid {theme.palette.primary.main}'
        },
        iconTheme: {
          primary: theme.palette.success.main,
          secondary: theme.palette.success.contrastText
        }
      })

      setLoadingRows(prevLoadingRows => prevLoadingRows.filter(index => index !== rowIndex))
    } else {
      setErrors({
        status: true,
        data: response,
        message: response.data.message
          ? response.data.message
          : response.data.error
          ? response.data.error
          : 'Gagal menghubungkan ke server'
      })

      setLoadingRows(prevLoadingRows => prevLoadingRows.filter(index => index !== rowIndex))
    }

    await getList(page, rowsPerPage)
  }

  const RowColumn = props => {
    let value = props.row[props.column.id]

    switch (props.column.id) {
      case 'action':
        if (props.loading) {
          value = (
            <div style={{ display: 'flex' }}>
              <CircularProgress color='success' />
              <Fab
                color='primary'
                aria-label='detail'
                size='small'
                onClick={e => clickDetail(row)}
                style={{ marginLeft: '5px' }}
              >
                <Icon icon='tabler:eye' />
              </Fab>
            </div>
          )
        } else {
          value = (
            <div style={{ display: 'flex' }}>
              <Fab
                color='success'
                aria-label='reload'
                size='small'
                onClick={e => syncData(props.row, props.index)}
                disabled={props.row.is_send == 1 && props.row.is_send_email == 1}
              >
                <Icon icon='tabler:reload' />
              </Fab>
              <Fab
                color='primary'
                aria-label='detail'
                size='small'
                onClick={e => clickDetail(props.row)}
                style={{ marginLeft: '5px' }}
              >
                <Icon icon='tabler:eye' />
              </Fab>
            </div>
          )
        }

        break

      case 'no':
        value = props.index + 1
        break

      case 'is_send':
      case 'is_send_email':
        if (value == 1) {
          value = <Chip label='True' color='success' />
        } else {
          value = <Chip label='False' color='error' />
        }
        break

      case 'is_pk':
      case 'is_badan_hukum':
      case 'is_npwp':
        if (value == 1) {
          value = <Chip label='Ya' color='success' sx={{ width: '60px' }} />
        } else {
          value = <Chip label='Tidak' color='error' sx={{ width: '60px' }} />
        }
        break
    }

    return <TableCell align={props.column.align}>{value}</TableCell>
  }

  return (
    <>
      <div style={{ position: 'relative' }}>
        {isLoading ? (
          <div
            style={{
              position: 'absolute',
              top: 0,
              left: 0,
              width: '100%',
              height: '100%',
              display: 'flex',
              justifyContent: 'center',
              alignItems: 'center',
              backgroundColor: 'rgb(187 187 187 / 50%)',
              zIndex: 1100
            }}
          >
            <CircularProgress />
          </div>
        ) : (
          ''
        )}
        <div style={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', padding: '20px' }}>
          <div>
            <Button variant='contained' color='success' onClick={clickExport}>
              Export
            </Button>
          </div>
          <div>
            <CustomTextField
              label=''
              value={search}
              onChange={e => changeSearch(e)}
              onKeyPress={e => {
                if (e.key === 'Enter') {
                  getList(page, rowsPerPage)
                }
              }}
            />
            <Button variant='contained' onClick={clickSearch}>
              Search
            </Button>
          </div>
        </div>
        <TableContainer component={Paper} sx={{ position: 'relative', minHeight: '300px' }}>
          <Table stickyHeader aria-label='sticky table'>
            <TableHead>
              <TableRow>
                {columns.map((column, index) => (
                  <TableCell
                    key={`header-${column.id}-${index}`}
                    align={column.align}
                    sx={{ minWidth: column.minWidth }}
                  >
                    {column.label}
                  </TableCell>
                ))}
              </TableRow>
            </TableHead>
            <TableBody>
              {dataList.map((row, index) => {
                return (
                  <TableRow hover role='checkbox' tabIndex={-1} key={`row-${index}`}>
                    {columns.map((column, index2) => {
                      return (
                        <RowColumn
                          key={`row-${index2}-${column.id}-${index}`}
                          column={column}
                          row={row}
                          index={index}
                          loading={loadingRows.includes(index)}
                        />
                      )
                    })}
                  </TableRow>
                )
              })}
            </TableBody>
          </Table>
        </TableContainer>
        <TablePagination
          rowsPerPageOptions={[10, 25, 100]}
          component='div'
          count={responseList ? responseList.total : 0}
          rowsPerPage={rowsPerPage}
          page={page}
          onPageChange={handleChangePage}
          onRowsPerPageChange={handleChangeRowsPerPage}
        />
      </div>
    </>
  )
}

export default TableData
