// ** React Imports
import { useState, useEffect, useRef } from 'react'

// ** Next Imports
import Link from 'next/link'
import axios from 'axios'
import ReCAPTCHA from 'react-google-recaptcha'

// ** MUI Components
import Alert from '@mui/material/Alert'
import Button from '@mui/material/Button'
import Divider from '@mui/material/Divider'
import Checkbox from '@mui/material/Checkbox'
import Typography from '@mui/material/Typography'
import IconButton from '@mui/material/IconButton'
import Box from '@mui/material/Box'
import useMediaQuery from '@mui/material/useMediaQuery'
import { styled, useTheme } from '@mui/material/styles'
import InputAdornment from '@mui/material/InputAdornment'
import MuiFormControlLabel from '@mui/material/FormControlLabel'
import CardContent from '@mui/material/CardContent'
import MuiCard from '@mui/material/Card'
import toast from 'react-hot-toast'

// ** Custom Component Import
import CustomTextField from 'src/@core/components/mui/text-field'

// ** Icon Imports
import Icon from 'src/@core/components/icon'

// ** Third Party Imports
import * as yup from 'yup'
import { useForm, Controller } from 'react-hook-form'
import { yupResolver } from '@hookform/resolvers/yup'

// ** Hooks
import { useAuth } from 'src/hooks/useAuth'
import useBgColor from 'src/@core/hooks/useBgColor'
import { useSettings } from 'src/@core/hooks/useSettings'

// ** Configs
import themeConfig from 'src/configs/themeConfig'

// ** Layout Import
import BlankLayout from 'src/@core/layouts/BlankLayout'

// ** Demo Imports
import AuthIllustrationV1Wrapper from 'src/views/pages/auth/AuthIllustrationV1Wrapper'

// PGN
import logoPGN from 'public/images/pgn.png'
import { replace } from 'stylis'

const Card = styled(MuiCard)(({ theme }) => ({
  [theme.breakpoints.up('sm')]: { width: '25rem' }
}))

const LinkStyled = styled(Link)(({ theme }) => ({
  textDecoration: 'none',
  color: `${theme.palette.primary.main} !important`
}))

const FormControlLabel = styled(MuiFormControlLabel)(({ theme }) => ({
  '& .MuiFormControlLabel-label': {
    color: theme.palette.text.secondary
  }
}))

const schema = yup.object().shape({
  email: yup.string().email().required(),
  password: yup.string().min(5).required()
})

const defaultValues = {
  password: '',
  email: ''
}

const LoginPage = () => {
  const [rememberMe, setRememberMe] = useState(true)
  const [showPassword, setShowPassword] = useState(false)

  const [email, setEmail] = useState('')
  const [password, setPassword] = useState('')
  const recaptchaRef = useRef(null)
  const [captcha, setCaptcha] = useState(null)

  const [errors, setErrors] = useState({
    status: false,
    data: {},
    message: ''
  })
  const [isLoading, setIsLoading] = useState(false)

  // ** Hooks
  const auth = useAuth()
  const theme = useTheme()
  const bgColors = useBgColor()
  const { settings } = useSettings()
  const hidden = useMediaQuery(theme.breakpoints.down('md'))

  useEffect(() => {
    if (errors.status) {
      toast.error(errors.message)
      setErrors({
        status: false,
        data: {},
        message: ''
      })
    }
  }, [errors])

  const submitLogin = async () => {
    if (email == '') {
      return toast.error('Email belum diisi')
    }

    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/
    if (!emailRegex.test(email)) {
      return toast.error('Format email tidak valid')
    }

    if (password == '') {
      return toast.error('Password belum diisi')
    }

    if (!captcha) {
      return toast.error('Pastikan Captcha Diisi')
    }

    // console.log('captcha = ', captcha)

    // CEK CAPTCHA

    // try {
    //   let secretKey = process.env.NEXT_PUBLIC_CAPTCHA_SECRET_KEY
    //   let verificationURL = process.env.NEXT_PUBLIC_CAPTCHA_VERIFICATION_URL
    //   verificationURL = verificationURL.replaceAll('{secretKey}', secretKey).replaceAll('{token}', captcha)

    //   let headers = {
    //     'Access-Control-Allow-Origin': '*',
    //     'Access-Control-Allow-Methods': 'POST'
    //   }

    //   const responseCaptcha = await axios.post(verificationURL, headers)
    //   const { success } = responseCaptcha.data

    //   console.log('success = ', success)

    //   return success
    // } catch (error) {
    //   console.error('Error verifying reCAPTCHA:', error)

    //   return false
    // }

    // END CEK CAPTCHA

    setIsLoading(true)

    await auth.login({ email, password, rememberMe }, response => {
      setIsLoading(false)
      setErrors({
        status: true,
        data: response,
        message: response.message
      })
    })

    handleRefresh()
  }

  const handleKeyPress = e => {
    if (e.key === 'Enter') {
      submitLogin()
    }
  }

  const handleRefresh = () => {
    recaptchaRef.current.reset()

    setCaptcha(null)
  }

  return (
    <Box className='content-center' sx={{}}>
      <AuthIllustrationV1Wrapper>
        <Card>
          <CardContent sx={{ p: theme => `${theme.spacing(10.5, 8, 8)} !important` }}>
            <Box sx={{ mb: 5, display: 'flex', justifyContent: 'center' }}>
              <img src={logoPGN.src} alt='' width={100} height={100} />
            </Box>
            <Box sx={{ display: 'flex', alignItems: 'center', justifyContent: 'center' }}>
              <Typography variant='h4' sx={{ ml: 2.5, fontWeight: 500, textAlign: 'center' }}>
                {themeConfig.templateName}
              </Typography>
            </Box>
            <Box sx={{ mb: 8, display: 'flex', alignItems: 'center', justifyContent: 'center' }}>
              <Typography variant='h4' sx={{ ml: 2.5, fontWeight: 500, textAlign: 'center' }}>
                Login
              </Typography>
            </Box>
            <CustomTextField
              autoFocus
              fullWidth
              value={email}
              onChange={e => setEmail(e.target.value)}
              id='email'
              label='Email'
              sx={{ mb: 4 }}
              placeholder='name@email.com'
            />

            <CustomTextField
              fullWidth
              value={password}
              onChange={e => setPassword(e.target.value)}
              onKeyPress={e => handleKeyPress(e)}
              sx={{ mb: 1.5 }}
              label='Password'
              id='password'
              placeholder='············'
              type={showPassword ? 'text' : 'password'}
              InputProps={{
                endAdornment: (
                  <InputAdornment position='end'>
                    <IconButton
                      edge='end'
                      onMouseDown={e => e.preventDefault()}
                      onClick={() => setShowPassword(!showPassword)}
                      aria-label='toggle password visibility'
                    >
                      <Icon fontSize='1.25rem' icon={showPassword ? 'tabler:eye' : 'tabler:eye-off'} />
                    </IconButton>
                  </InputAdornment>
                )
              }}
            />

            {/* CAPTCHA */}
            <div style={{ display: 'flex', justifyContent: 'center' }}>
              <ReCAPTCHA sitekey={process.env.NEXT_PUBLIC_CAPTCHA_SITE_KEY} ref={recaptchaRef} onChange={setCaptcha} />
            </div>
            {/* END CAPTCHA */}

            {/* <Box
                sx={{
                  mb: 1.75,
                  display: 'flex',
                  flexWrap: 'wrap',
                  alignItems: 'center',
                  justifyContent: 'space-between'
                }}
              >
                <FormControlLabel control={<Checkbox />} label='Remember Me' />
                <Typography component={LinkStyled} href='/pages/auth/forgot-password-v1'>
                  Forgot Password?
                </Typography>
              </Box> */}
            {isLoading ? (
              <Button fullWidth type='submit' variant='contained' sx={{ mb: 4, mt: 5 }} disabled>
                Loading...
              </Button>
            ) : (
              <Button fullWidth type='submit' variant='contained' sx={{ mb: 4, mt: 5 }} onClick={submitLogin}>
                Login
              </Button>
            )}
          </CardContent>
        </Card>
      </AuthIllustrationV1Wrapper>
    </Box>
  )
}
LoginPage.getLayout = page => <BlankLayout>{page}</BlankLayout>
LoginPage.guestGuard = true

export default LoginPage
