// ** React Imports
import { useState, useEffect, forwardRef, Fragment } from 'react'
import { useTheme } from '@mui/material/styles'

// ** MUI Imports
import Paper from '@mui/material/Paper'
import Grid from '@mui/material/Grid'
import Table from '@mui/material/Table'
import TableRow from '@mui/material/TableRow'
import TableHead from '@mui/material/TableHead'
import TableBody from '@mui/material/TableBody'
import TableCell from '@mui/material/TableCell'
import TableContainer from '@mui/material/TableContainer'
import TablePagination from '@mui/material/TablePagination'
import Button from '@mui/material/Button'
import ButtonGroup from '@mui/material/ButtonGroup'
import CustomTextField from 'src/@core/components/mui/text-field'
import toast from 'react-hot-toast'
import CircularProgress from '@mui/material/CircularProgress'
import Fab from '@mui/material/Fab'
import Icon from 'src/@core/components/icon'
import { useRouter } from 'next/router'
import DatePicker from 'react-datepicker'
import DatePickerWrapper from 'src/@core/styles/libs/react-datepicker'
import CustomAutocomplete from 'src/@core/components/mui/autocomplete'

// PGN
import { endPoint } from 'src/service/constans'
import { callGet, callPost, callExport } from 'src/service/api'

const columns = [
  { id: 'no', label: 'No', minWidth: 50 },
  {
    id: 'action',
    label: 'Action',
    minWidth: 100,
    align: 'center'
  },
  { id: 'form_id', label: 'Registrasi Id', minWidth: 170 },
  { id: 'nama', label: 'Nama Pelanggan', minWidth: 170 },
  { id: '', label: 'No ID Pelanggan', minWidth: 100 },
  { id: 'town_name', label: 'Kota/Kabupaten', minWidth: 100 },
  { id: '', label: 'NPWP Pelanggan', minWidth: 100 },
  { id: 'ktp', label: 'NIK Pelanggan', minWidth: 100 },
  { id: 'nama', label: 'Nama Sesuai NIK/NPWP', minWidth: 100 },
  { id: 'alamat', label: 'Alamat NIK/NPWP', minWidth: 100 },
  { id: 'no_telpon', label: 'HP', minWidth: 100 },
  { id: 'email', label: 'Email', minWidth: 100 }
]

const TableData = () => {
  const router = useRouter()
  const theme = useTheme()

  // ** States
  const [page, setPage] = useState(0)
  const [rowsPerPage, setRowsPerPage] = useState(10)
  const [search, setSearch] = useState('')
  const [isLoading, setIsLoading] = useState(false)
  const [responseList, setResponseList] = useState(null)
  const [dataList, setDataList] = useState([])
  const [date, setDate] = useState(new Date())

  const [openArea, setOpenArea] = useState(false)
  const [optionsArea, setOptionsArea] = useState([])
  const [loadingArea, setLoadingArea] = useState(false)

  const handleChangePage = async (event, newPage) => {
    setPage(newPage)
    await getList(newPage, rowsPerPage)
  }

  const handleChangeRowsPerPage = async event => {
    setRowsPerPage(+event.target.value)
    setPage(0)
    getList(0, +event.target.value)
  }

  // PGN
  const [errors, setErrors] = useState({
    status: false,
    data: {},
    message: ''
  })

  useEffect(() => {
    if (errors.status) {
      if (errors.message == 'Failed validation' && errors.data.data.errors) {
        for (let field in errors.data.data.errors) {
          let errorMessages = errors.data.data.errors[field]
          errorMessages.forEach(errorMessage => {
            toast.error(errorMessage, {
              duration: 6000,
              style: {
                padding: '16px',
                color: theme.palette.error.main,
                border: '1px solid {theme.palette.error.main}'
              },
              iconTheme: {
                primary: theme.palette.error.main,
                secondary: theme.palette.error.contrastText
              }
            })
          })
        }
      } else {
        toast.error(errors.message, {
          duration: 6000,
          style: {
            padding: '16px',
            color: theme.palette.error.main,
            border: '1px solid {theme.palette.error.main}'
          },
          iconTheme: {
            primary: theme.palette.error.main,
            secondary: theme.palette.error.contrastText
          }
        })
      }

      setErrors({
        status: false,
        data: {},
        message: ''
      })
    }
  }, [errors])

  useEffect(() => {
    getList(page, rowsPerPage)
  }, [])

  useEffect(() => {
    let active = true
    if (!loadingArea) {
      return undefined
    }

    setErrors({
      data: {},
      message: '',
      status: false
    })

    if (optionsArea.length == 0) {
      fetchDataArea(active)
    } else {
      setLoadingArea(false)
    }

    return () => {
      active = false
    }
  }, [loadingArea])

  const fetchDataArea = async active => {
    const responseArea = await callGet(endPoint.getAreaRegistrationReport)

    if (!responseArea.status) {
      setErrors({
        data: responseArea.data,
        message: responseArea.message,
        status: true
      })
    } else {
      if (active) {
        setOptionsArea(responseArea.data.data)
      }
    }

    setLoadingArea(false)
  }

  const handleSelectArea = (event, newValue) => {
    setSearch(newValue)
  }

  const getList = async (temp_page, temp_rowsPerPage) => {
    setIsLoading(true)

    let params = {
      perPage: temp_rowsPerPage,
      page: temp_page + 1
    }
    if (search != '' && search != null) {
      params.search = search.id ? search.id : ''
    }
    if (date) {
      params.date = date
    }
    let response = await callGet(endPoint.getListRegistrationReport, params)

    if (response.status) {
      setResponseList(response.data.data)
      setDataList(response.data.data.data)

      setIsLoading(false)
    } else {
      setResponseList(null)
      setDataList([])
      setErrors({
        status: true,
        data: response,
        message: response.data.message
          ? response.data.message
          : response.data.error
          ? response.data.error
          : 'Gagal menghubungkan ke server'
      })

      setIsLoading(false)
    }
  }

  const changeSearch = e => {
    setSearch(e.target.value)
  }

  const clickSearch = () => {
    getList(page, rowsPerPage)
  }

  const clickDetail = row => {
    router.push(`/register-detail/${row.id}`)
  }

  const clickExport = async () => {
    setIsLoading(true)

    let params = {}
    if (search != '' && search != null) {
      params.search = search.id ? search.id : ''
    }
    if (date) {
      params.date = date
    }
    let response = await callExport(endPoint.getExportRegistrationReport, params)

    if (response.status) {
      console.log('response ', response)
      const url = window.URL.createObjectURL(new Blob([response.data]))

      const currentDate = new Date()
      const dateString = currentDate.toISOString().split('T')[0] // Ambil bagian tanggal dari timestamp ISO
      const fileName = `data_registrasi_report_${dateString}.xlsx`

      const link = document.createElement('a')
      link.href = url
      link.setAttribute('download', fileName)
      document.body.appendChild(link)
      link.click()

      window.URL.revokeObjectURL(url)

      setIsLoading(false)
    } else {
      setIsLoading(false)
      setErrors({
        status: true,
        data: response,
        message: response.data.message
          ? response.data.message
          : response.data.error
          ? response.data.error
          : 'Gagal menghubungkan ke server'
      })
    }
  }

  const clickSendEmail = async () => {
    setIsLoading(true)

    let paylaod = {}

    if (search != '' && search != null) {
      paylaod.area_id = search.id ? search.id : ''
    }
    if (date) {
      paylaod.date = date
    }
    let response = await callPost(endPoint.postSendEmailRegistrationReport, paylaod)

    if (response.status) {
      toast.success(response.message, {
        duration: 6000,
        style: {
          padding: '16px',
          color: theme.palette.success.main,
          border: '1px solid {theme.palette.error.main}'
        },
        iconTheme: {
          primary: theme.palette.success.main,
          secondary: theme.palette.success.contrastText
        }
      })

      setIsLoading(false)
    } else {
      setErrors({
        status: true,
        data: response,
        message: response.data.message
          ? response.data.message
          : response.data.error
          ? response.data.error
          : 'Gagal menghubungkan ke server'
      })

      setIsLoading(false)
    }
  }

  const RowColumn = props => {
    let value = props.row[props.column.id]

    switch (props.column.id) {
      case 'action':
        value = (
          <div style={{ display: 'flex' }}>
            <Fab
              color='primary'
              aria-label='detail'
              size='small'
              onClick={e => clickDetail(props.row)}
              style={{ marginLeft: '5px' }}
            >
              <Icon icon='tabler:eye' />
            </Fab>
          </div>
        )

        break

      case 'no':
        value = props.index + 1
        break

      case 'is_send':
        if (value == 1) {
          value = 'True'
        } else {
          value = 'False'
        }
        break
    }

    return <TableCell align={props.column.align}>{value}</TableCell>
  }

  return (
    <>
      <div style={{ position: 'relative' }}>
        {isLoading ? (
          <div
            style={{
              position: 'absolute',
              top: 0,
              left: 0,
              width: '100%',
              height: '100%',
              display: 'flex',
              justifyContent: 'center',
              alignItems: 'center',
              backgroundColor: 'rgb(187 187 187 / 50%)',
              zIndex: 1100
            }}
          >
            <CircularProgress />
          </div>
        ) : (
          ''
        )}
        <Grid container spacing={5}>
          <Grid item xs={12} sm={4}>
            <div style={{ display: 'flex', justifyContent: 'start', alignItems: 'center', padding: '20px' }}>
              <div>
                <Button variant='contained' color='success' onClick={e => clickExport()}>
                  Download
                </Button>
              </div>
              <div style={{ marginLeft: '10px' }}>
                <Button variant='contained' color='success' onClick={e => clickSendEmail()}>
                  Send Mail
                </Button>
              </div>
            </div>
          </Grid>
          <Grid item xs={12} sm={8}>
            <Grid
              container
              spacing={5}
              style={{
                display: 'flex',
                justifyContent: 'space-between',
                alignItems: 'center',
                padding: '20px',
                flexWrap: 'wrap',
                flexDirection: 'row'
              }}
            >
              <Grid item xs={12} sm={5} style={{ display: 'flex', justifyContent: 'end', alignItems: 'center' }}>
                <DatePickerWrapper>
                  <DatePicker
                    selected={date}
                    id='picker-clear'
                    popperPlacement='bottom-start'
                    onChange={newDate => {
                      if (newDate) {
                        // Mengambil jam dan menit dari tanggal sebelumnya
                        const dateNow = new Date()
                        const hours = dateNow.getHours()
                        const minutes = dateNow.getMinutes()

                        // Mengatur tanggal baru dengan waktu yang sama
                        newDate.setHours(hours)
                        newDate.setMinutes(minutes)

                        // Mengatur tanggal baru dengan waktu yang sama
                        console.log(newDate)
                      }
                      setDate(newDate)
                    }}
                    placeholderText='Click to select a date'
                    customInput={<CustomTextField label='' />}
                    dateFormat='dd/MM/yyyy'
                  />
                </DatePickerWrapper>
              </Grid>
              <Grid item xs={12} sm={5} style={{ display: 'flex', justifyContent: 'end', alignItems: 'center' }}>
                <CustomAutocomplete
                  value={search}
                  open={openArea}
                  options={optionsArea.map(option => ({ ...option, key: option.id }))} // Menambahkan properti key dengan menggunakan id sebagai nilai unik
                  loading={loadingArea}
                  onOpen={() => {
                    setOpenArea(true)
                    setLoadingArea(true)
                  }}
                  onClose={() => {
                    setOpenArea(false)
                    setLoadingArea(false)
                  }}
                  onChange={handleSelectArea}
                  id='select-area'
                  getOptionLabel={option => option.name || ''}
                  isOptionEqualToValue={(option, value) => option.id === value.id}
                  sx={{ width: '100%' }}
                  renderInput={params => (
                    <CustomTextField
                      {...params}
                      label=''
                      InputProps={{
                        ...params.InputProps,
                        endAdornment: (
                          <Fragment>
                            {loadingArea ? <CircularProgress size={20} /> : null}
                            {params.InputProps.endAdornment}
                          </Fragment>
                        )
                      }}
                      placeholder='Pilih Area'
                    />
                  )}
                />
              </Grid>
              <Grid item xs={12} sm={2} style={{ display: 'flex', justifyContent: 'end', alignItems: 'center' }}>
                <Button variant='contained' onClick={clickSearch}>
                  Search
                </Button>
              </Grid>
            </Grid>
          </Grid>
        </Grid>
        <TableContainer component={Paper} sx={{ position: 'relative', minHeight: '300px' }}>
          <Table stickyHeader aria-label='sticky table'>
            <TableHead>
              <TableRow>
                {columns.map((column, index) => (
                  <TableCell
                    key={`header-${column.id}-${index}`}
                    align={column.align}
                    sx={{ minWidth: column.minWidth }}
                  >
                    {column.label}
                  </TableCell>
                ))}
              </TableRow>
            </TableHead>
            <TableBody>
              {dataList.map((row, index) => {
                return (
                  <TableRow hover role='checkbox' tabIndex={-1} key={`row-${index}`}>
                    {columns.map((column, index2) => {
                      return (
                        <RowColumn
                          key={`row-${index2}-${column.id}-${index}`}
                          column={column}
                          row={row}
                          index={index}
                        />
                      )
                    })}
                  </TableRow>
                )
              })}
            </TableBody>
          </Table>
        </TableContainer>
        <TablePagination
          rowsPerPageOptions={[10, 25, 100]}
          component='div'
          count={responseList ? responseList.total : 0}
          rowsPerPage={rowsPerPage}
          page={page}
          onPageChange={handleChangePage}
          onRowsPerPageChange={handleChangeRowsPerPage}
        />
      </div>
    </>
  )
}

export default TableData
