// ** Next Import
import { useState, useEffect } from 'react'
import { useRouter } from 'next/router'
import { useTheme } from '@mui/material/styles'

// ** MUI Imports
import Grid from '@mui/material/Grid'
import Card from '@mui/material/Card'
import Typography from '@mui/material/Typography'
import CardHeader from '@mui/material/CardHeader'
import CustomTextField from 'src/@core/components/mui/text-field'
import CardContent from '@mui/material/CardContent'
import toast from 'react-hot-toast'
import Button from '@mui/material/Button'
import Checkbox from '@mui/material/Checkbox'
import FormLabel from '@mui/material/FormLabel'
import CardActions from '@mui/material/CardActions'
import CircularProgress from '@mui/material/CircularProgress'

import Divider from '@mui/material/Divider'

// ** Custom Components Imports
import PageHeader from 'src/@core/components/page-header'

// PGN
import { endPoint } from 'src/service/constans'
import { callPost } from 'src/service/api'

const UserCreate = () => {
  const theme = useTheme()
  const router = useRouter()
  const { id } = router.query

  const [dataUser, setDataUser] = useState({
    name: '',
    email: '',
    is_active: false
  })
  const [isLoading, setIsLoading] = useState(false)

  const setStateLocal = (key, value) => {
    setDataUser(prevState => ({
      ...prevState,
      [key]: value
    }))
  }

  // ERROR
  const [errors, setErrors] = useState({
    status: false,
    data: {},
    message: ''
  })

  useEffect(() => {
    if (errors.status) {
      if (errors.message == 'Failed validation' && errors.data.data.errors) {
        for (let field in errors.data.data.errors) {
          let errorMessages = errors.data.data.errors[field]
          errorMessages.forEach(errorMessage => {
            toast.error(errorMessage, {
              duration: 6000,
              style: {
                padding: '16px',
                color: theme.palette.error.main,
                border: '1px solid {theme.palette.error.main}'
              },
              iconTheme: {
                primary: theme.palette.error.main,
                secondary: theme.palette.error.contrastText
              }
            })
          })
        }
      } else {
        toast.error(errors.message, {
          duration: 6000,
          style: {
            padding: '16px',
            color: theme.palette.error.main,
            border: '1px solid {theme.palette.error.main}'
          },
          iconTheme: {
            primary: theme.palette.error.main,
            secondary: theme.palette.error.contrastText
          }
        })
      }

      setErrors({
        status: false,
        data: {},
        message: ''
      })
    }
  }, [errors])

  // END ERROR

  const clickCancel = () => {
    router.push(`/user`)
  }

  const handleSave = async () => {
    if (dataUser.name == '') {
      setErrors({
        status: true,
        data: null,
        message: 'User Name belum diisi.'
      })

      return
    }

    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/
    if (!emailRegex.test(dataUser.email)) {
      setErrors({
        status: true,
        data: null,
        message: 'Email harus dalam format yang valid.'
      })

      return
    }

    setIsLoading(true)

    let payload = {
      name: dataUser.name,
      email: dataUser.email,
      is_active: dataUser.is_active ? 1 : 0
    }

    let response = await callPost(`${endPoint.getUser}`, payload)

    if (response.status) {
      setDataUser({
        name: '',
        email: '',
        is_active: false
      })

      toast.success(response.message, {
        duration: 6000,
        style: {
          padding: '16px',
          color: theme.palette.success.main,
          border: '1px solid {theme.palette.primary.main}'
        },
        iconTheme: {
          primary: theme.palette.success.main,
          secondary: theme.palette.success.contrastText
        }
      })

      router.push('/user')
    } else {
      setErrors({
        status: true,
        data: response,
        message: response.data.message
          ? response.data.message
          : response.data.error
          ? response.data.error
          : 'Gagal menghubungkan ke server'
      })
    }

    setIsLoading(false)
  }

  return (
    <Grid container spacing={6}>
      <PageHeader title={<Typography variant='h4'>User Management - Create</Typography>} />
      <Grid item xs={12}>
        <div style={{ position: 'relative' }}>
          {isLoading && (
            <div
              style={{
                position: 'absolute',
                top: 0,
                left: 0,
                width: '100%',
                height: '100%',
                display: 'flex',
                justifyContent: 'center',
                alignItems: 'center',
                backgroundColor: 'rgb(187 187 187 / 50%)',
                zIndex: 1100
              }}
            >
              <CircularProgress />
            </div>
          )}
          <Card>
            <CardHeader title='User Data' sx={{ fontSize: '13pt', padding: '16px' }} />
            <Divider sx={{ m: '0 !important' }} />
            <CardContent sx={{ p: theme => `${theme.spacing(3, 5.25, 4)} !important` }}>
              <Grid container spacing={5} sx={{ marginBottom: '15px' }}>
                <Grid item xs={12} sm={6}>
                  <CustomTextField
                    fullWidth
                    required
                    label='User Name'
                    value={dataUser.name}
                    onChange={e => setStateLocal('name', e.target.value)}
                  />
                </Grid>
              </Grid>
              <Grid container spacing={5} sx={{ marginBottom: '15px' }}>
                <Grid item xs={12} sm={6}>
                  <CustomTextField
                    fullWidth
                    required
                    label='Email'
                    value={dataUser.email}
                    onChange={e => setStateLocal('email', e.target.value)}
                  />
                </Grid>
              </Grid>
              <Grid container spacing={5} sx={{ marginBottom: '15px' }}>
                <Grid item xs={12} sm={6}>
                  <FormLabel style={{ fontSize: '10pt' }}>Is Active</FormLabel>
                  <br />
                  <Checkbox checked={dataUser.is_active} onChange={e => setStateLocal('is_active', e.target.checked)} />
                </Grid>
              </Grid>
            </CardContent>
            <Divider sx={{ m: '0 !important' }} />
            <CardActions sx={{ display: 'flex', justifyContent: 'end' }}>
              <Button sx={{ mr: 2 }} color='primary' variant='contained' onClick={handleSave}>
                Save
              </Button>
              <Button color='secondary' variant='contained' onClick={clickCancel}>
                Cancel
              </Button>
            </CardActions>
          </Card>
        </div>
      </Grid>
    </Grid>
  )
}

export default UserCreate
