// ** React Imports
import { useState, useEffect, Fragment } from 'react'
import { useTheme } from '@mui/material/styles'

// ** MUI Imports
import Paper from '@mui/material/Paper'
import Table from '@mui/material/Table'
import TableRow from '@mui/material/TableRow'
import TableHead from '@mui/material/TableHead'
import TableBody from '@mui/material/TableBody'
import TableCell from '@mui/material/TableCell'
import TableContainer from '@mui/material/TableContainer'
import TablePagination from '@mui/material/TablePagination'
import Button from '@mui/material/Button'
import ButtonGroup from '@mui/material/ButtonGroup'
import CustomTextField from 'src/@core/components/mui/text-field'
import toast from 'react-hot-toast'
import CircularProgress from '@mui/material/CircularProgress'
import Fab from '@mui/material/Fab'
import Icon from 'src/@core/components/icon'
import { useRouter } from 'next/router'
import Checkbox from '@mui/material/Checkbox'
import FormControlLabel from '@mui/material/FormControlLabel'
import Dialog from '@mui/material/Dialog'
import DialogTitle from '@mui/material/DialogTitle'
import DialogContent from '@mui/material/DialogContent'
import DialogActions from '@mui/material/DialogActions'
import DialogContentText from '@mui/material/DialogContentText'

// PGN
import { endPoint } from 'src/service/constans'
import { callGet, callPost, callExport } from 'src/service/api'

const columns = [
  { id: 'no', label: 'No', width: 20 },
  { id: 'name', label: 'User Name', minWidth: 170 },
  { id: 'email', label: 'Email', minWidth: 100 },
  { id: 'is_active', label: 'Is Active', minWidth: 100 },
  {
    id: 'action',
    label: 'Action',
    minWidth: 100,
    align: 'center'
  }
]

const TableData = () => {
  const router = useRouter()
  const theme = useTheme()

  // ** States
  const [page, setPage] = useState(0)
  const [rowsPerPage, setRowsPerPage] = useState(10)
  const [search, setSearch] = useState('')
  const [isLoading, setIsLoading] = useState(false)
  const [responseList, setResponseList] = useState(null)
  const [dataList, setDataList] = useState([])

  const [dataDelete, setDataDelete] = useState(null)
  const [openDelete, setOpenDelete] = useState(false)
  const handleClickOpenDelete = () => setOpenDelete(true)

  const handleCloseDelete = () => {
    setDataDelete(null)
    setOpenDelete(false)
  }

  const handleChangePage = async (event, newPage) => {
    setPage(newPage)
    await getList(newPage, rowsPerPage)
  }

  const handleChangeRowsPerPage = async event => {
    setRowsPerPage(+event.target.value)
    setPage(0)
    getList(0, +event.target.value)
  }

  // PGN
  const [errors, setErrors] = useState({
    status: false,
    data: {},
    message: ''
  })

  useEffect(() => {
    if (errors.status) {
      if (errors.message == 'Failed validation' && errors.data.data.errors) {
        for (let field in errors.data.data.errors) {
          let errorMessages = errors.data.data.errors[field]
          errorMessages.forEach(errorMessage => {
            toast.error(errorMessage, {
              duration: 6000,
              style: {
                padding: '16px',
                color: theme.palette.error.main,
                border: '1px solid {theme.palette.error.main}'
              },
              iconTheme: {
                primary: theme.palette.error.main,
                secondary: theme.palette.error.contrastText
              }
            })
          })
        }
      } else {
        toast.error(errors.message, {
          duration: 6000,
          style: {
            padding: '16px',
            color: theme.palette.error.main,
            border: '1px solid {theme.palette.error.main}'
          },
          iconTheme: {
            primary: theme.palette.error.main,
            secondary: theme.palette.error.contrastText
          }
        })
      }

      setErrors({
        status: false,
        data: {},
        message: ''
      })
    }
  }, [errors])

  useEffect(() => {
    getList(page, rowsPerPage)
  }, [])

  const getList = async (temp_page, temp_rowsPerPage) => {
    setIsLoading(true)

    let params = {
      perPage: temp_rowsPerPage,
      page: temp_page + 1
    }
    if (search != '') {
      params.search = search
    }
    let response = await callGet(endPoint.getUser, params)

    if (response.status) {
      setResponseList(response.data.data)
      setDataList(response.data.data.data)

      setIsLoading(false)
    } else {
      setResponseList(null)
      setDataList([])
      setErrors({
        status: true,
        data: response,
        message: response.data.message
          ? response.data.message
          : response.data.error
          ? response.data.error
          : 'Gagal menghubungkan ke server'
      })

      setIsLoading(false)
    }
  }

  const changeSearch = e => {
    setSearch(e.target.value)
  }

  const clickSearch = () => {
    getList(page, rowsPerPage)
  }

  const clickCreate = () => {
    router.push(`/user/create`)
  }

  const clickEdit = row => {
    router.push(`/user/${row.id}`)
  }

  const clickDelete = row => {
    setDataDelete(row)
    setOpenDelete(true)
  }

  const handleDelete = async () => {
    setIsLoading(true)

    let params = {
      id: dataDelete ? dataDelete.id : '',
      name: dataDelete ? dataDelete.name : '',
      email: dataDelete ? dataDelete.email : ''
    }

    let response = await callPost(endPoint.deleteUser, params)

    if (response.status) {
      toast.success(response.message, {
        duration: 6000,
        style: {
          padding: '16px',
          color: theme.palette.success.main,
          border: '1px solid {theme.palette.primary.main}'
        },
        iconTheme: {
          primary: theme.palette.success.main,
          secondary: theme.palette.success.contrastText
        }
      })
    } else {
      setErrors({
        status: true,
        data: response,
        message: response.data.message
          ? response.data.message
          : response.data.error
          ? response.data.error
          : 'Gagal menghubungkan ke server'
      })
    }
    setIsLoading(false)
    handleCloseDelete()
    getList(page, rowsPerPage)
  }

  const DialogDelete = () => {
    return (
      <Fragment>
        <Dialog
          open={openDelete}
          disableEscapeKeyDown
          aria-labelledby='alert-dialog-title'
          aria-describedby='alert-dialog-description'
          onClose={(event, reason) => {
            if (reason !== 'backdropClick') {
              handleCloseDelete()
            }
          }}
        >
          <DialogTitle id='alert-dialog-title'>
            <b>Apakah anda yakin ingin menghapus data berikut?</b>
          </DialogTitle>
          <DialogContent>
            <DialogContentText id='alert-dialog-description'>
              <ul>
                <li>
                  User Name : <b>{dataDelete ? dataDelete.name : ''}</b>
                </li>
                <li>
                  Email : <b>{dataDelete ? dataDelete.email : ''}</b>
                </li>
              </ul>
            </DialogContentText>
          </DialogContent>
          <DialogActions className='dialog-actions-dense' style={{ padding: '20px' }}>
            <Button variant='contained' color='error' onClick={handleDelete}>
              Ya, Hapus
            </Button>
            <Button variant='contained' color='secondary' onClick={handleCloseDelete}>
              Batal
            </Button>
          </DialogActions>
        </Dialog>
      </Fragment>
    )
  }

  return (
    <>
      <div style={{ position: 'relative' }}>
        {isLoading ? (
          <div
            style={{
              position: 'absolute',
              top: 0,
              left: 0,
              width: '100%',
              height: '100%',
              display: 'flex',
              justifyContent: 'center',
              alignItems: 'center',
              backgroundColor: 'rgb(187 187 187 / 50%)',
              zIndex: 1100
            }}
          >
            <CircularProgress />
          </div>
        ) : (
          ''
        )}
        <div style={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', padding: '20px' }}>
          <div>
            <Button variant='contained' color='success' onClick={clickCreate}>
              Create
            </Button>
          </div>
          <div>
            <CustomTextField
              label=''
              value={search}
              onChange={e => changeSearch(e)}
              onKeyPress={e => {
                if (e.key === 'Enter') {
                  getList(page, rowsPerPage)
                }
              }}
            />
            <Button variant='contained' onClick={clickSearch}>
              Search
            </Button>
          </div>
        </div>
        <TableContainer component={Paper} sx={{ position: 'relative', minHeight: '300px' }}>
          <Table stickyHeader aria-label='sticky table'>
            <TableHead>
              <TableRow>
                {columns.map(column => (
                  <TableCell key={column.id} align={column.align} sx={{ minWidth: column.minWidth }}>
                    {column.label}
                  </TableCell>
                ))}
              </TableRow>
            </TableHead>
            <TableBody>
              {dataList.map((row, index) => {
                return (
                  <TableRow hover role='checkbox' tabIndex={-1} key={`row-${index}`}>
                    {columns.map(column => {
                      const value = column.id != 'no' ? row[column.id] : index + 1

                      return (
                        <TableCell key={column.id} align={column.align}>
                          {column.id == 'action' ? (
                            <Fragment>
                              <Fab color='primary' aria-label='edit' size='small' onClick={e => clickEdit(row)}>
                                <Icon icon='tabler:pencil' />
                              </Fab>
                              <Fab
                                color='error'
                                aria-label='delete'
                                size='small'
                                style={{ marginLeft: '5px' }}
                                onClick={e => clickDelete(row)}
                              >
                                <Icon icon='tabler:trash' />
                              </Fab>
                            </Fragment>
                          ) : column.id == 'is_active' ? (
                            <Checkbox name='basic-checked' disabled checked={row[column.id] == 1} />
                          ) : (
                            value
                          )}
                        </TableCell>
                      )
                    })}
                  </TableRow>
                )
              })}
            </TableBody>
          </Table>
        </TableContainer>
        <TablePagination
          rowsPerPageOptions={[10, 25, 100]}
          component='div'
          count={responseList ? responseList.total : 0}
          rowsPerPage={rowsPerPage}
          page={page}
          onPageChange={handleChangePage}
          onRowsPerPageChange={handleChangeRowsPerPage}
        />

        {/* DIALOG */}
        <DialogDelete />
      </div>
    </>
  )
}

export default TableData
