'use strict';

const css_parse = require('../css/parse.cjs');
const css_parser_export = require('../css/parser/export.cjs');
const css_parser_tokens = require('../css/parser/tokens.cjs');
const css_parser_tree = require('../css/parser/tree.cjs');
const svg_parse = require('./parse.cjs');
require('../css/parser/error.cjs');
require('../css/parser/strings.cjs');
require('../css/parser/text.cjs');

async function parseSVGStyle(svg, callback) {
  return svg_parse.parseSVG(svg, async (item) => {
    const tagName = item.tagName;
    const $element = item.$element;
    if (tagName === "style") {
      const content = $element.text();
      if (typeof content !== "string") {
        $element.remove();
        return;
      }
      const tokens = css_parser_tokens.getTokens(content);
      if (!(tokens instanceof Array)) {
        throw new Error("Error parsing style");
      }
      let changed2 = false;
      const selectorStart = [];
      const newTokens = [];
      for (let i = 0; i < tokens.length; i++) {
        const token = tokens[i];
        switch (token.type) {
          case "selector":
          case "at-rule":
            selectorStart.push(newTokens.length);
            break;
          case "close":
            selectorStart.pop();
            break;
        }
        if (token.type !== "rule") {
          newTokens.push(token);
          continue;
        }
        const value = token.value;
        const selectorTokens = selectorStart.map((index) => newTokens[index]).filter((item2) => item2 !== null);
        let result = callback({
          type: "global",
          prop: token.prop,
          value,
          token,
          selectorTokens,
          selectors: selectorTokens.reduce(
            (prev, current) => {
              switch (current.type) {
                case "selector": {
                  return prev.concat(current.selectors);
                }
              }
              return prev;
            },
            []
          ),
          prevTokens: newTokens,
          nextTokens: tokens.slice(i + 1)
        });
        if (result instanceof Promise) {
          result = await result;
        }
        if (result !== void 0) {
          if (result !== value) {
            changed2 = true;
            token.value = result;
          }
          newTokens.push(token);
        } else {
          changed2 = true;
        }
      }
      if (!changed2) {
        return;
      }
      const tree = css_parser_tree.tokensTree(
        newTokens.filter((token) => token !== null)
      );
      if (!tree.length) {
        $element.remove();
        return;
      }
      const newContent = css_parser_export.tokensToString(tree);
      item.$element.text(newContent);
      return;
    }
    const attribs = item.element.attribs;
    if (attribs.style === void 0) {
      return;
    }
    const parsedStyle = css_parse.parseInlineStyle(attribs.style);
    if (parsedStyle === null) {
      $element.removeAttr("style");
      return;
    }
    const props = Object.keys(parsedStyle);
    let changed = false;
    for (let i = 0; i < props.length; i++) {
      const prop = props[i];
      const value = parsedStyle[prop];
      let result = callback({
        type: "inline",
        prop,
        value,
        item
      });
      if (result instanceof Promise) {
        result = await result;
      }
      if (result !== value) {
        changed = true;
        if (result === void 0) {
          delete parsedStyle[prop];
        } else {
          parsedStyle[prop] = result;
        }
      }
    }
    if (!changed) {
      return;
    }
    const newStyle = Object.keys(parsedStyle).map((key) => key + ":" + parsedStyle[key] + ";").join("");
    if (!newStyle.length) {
      $element.removeAttr("style");
    } else {
      $element.attr("style", newStyle);
    }
  });
}

exports.parseSVGStyle = parseSVGStyle;
