// ** React Imports
import { forwardRef, useState, Fragment, useEffect } from 'react'

// ** MUI Imports
import Grid from '@mui/material/Grid'
import Button from '@mui/material/Button'
import Typography from '@mui/material/Typography'
import CircularProgress from '@mui/material/CircularProgress'
import CustomTextField from 'src/@core/components/mui/text-field'
import CustomAutocomplete from 'src/@core/components/mui/autocomplete'
import { styled } from '@mui/material/styles'
import MuiInputLabel from '@mui/material/InputLabel'
import Radio from '@mui/material/Radio'
import RadioGroup from '@mui/material/RadioGroup'
import FormControlLabel from '@mui/material/FormControlLabel'
import Checkbox from '@mui/material/Checkbox'

// ** CleaveJS Imports
import CleaveWrapper from 'src/@core/styles/libs/react-cleave'
import Cleave from 'cleave.js/react'
import 'cleave.js/dist/addons/cleave-phone.us'

import { useDispatch, useSelector } from 'react-redux'
import { setErrorApi, setKelengkapan } from 'src/store/apps/kelengkapan-data'

import { callGet } from 'src/service/api'
import { Endpoint } from 'src/service/constant'

const InputLabel = styled(MuiInputLabel)(({ theme }) => ({
  lineHeight: 1.154,
  maxWidth: 'max-content',
  marginBottom: theme.spacing(1),
  color: theme.palette.text.primary,
  fontSize: theme.typography.body2.fontSize
}))

const VisuallyHiddenInput = styled('input')({
  clip: 'rect(0 0 0 0)',
  clipPath: 'inset(50%)',
  height: 1,
  overflow: 'hidden',
  position: 'absolute',
  bottom: 0,
  left: 0,
  whiteSpace: 'nowrap',
  width: 1
})

const FormKelengkapan = () => {
  const dispatch = useDispatch()
  const store = useSelector(state => state.kelengkapanData)

  const setDefaultErrorState = () => {
    dispatch(
      setErrorApi({
        data: {},
        message: '',
        status: false
      })
    )
  }

  const [dataKelengkapan, setDataKelengkapan] = useState({
    nama_pelanggan: '',
    customer_id: '',
    provinsi: '',
    kota: '',
    is_pk: '',
    is_badan_hukum: '',
    is_npwp: '',
    npwp: '',
    ktp: '',
    ktp_file: '',
    nama_ktp: '',
    alamat_ktp: '',
    alamat_lokasi_pemasangan: '',
    no_telpon: '',
    email: '',
    media_informasi: {
      sms: false,
      whatsapp: false,
      email: false,
      pgn_mobile: false
    }
  })

  useEffect(() => {
    setDataKelengkapan(store.dataKelengkapan)

    if (store.dataResponseVerifikasiId) {
      let temp_no_telpon = store.dataResponseVerifikasiId.data.CELL_PH_NUM

      if (temp_no_telpon && temp_no_telpon.length > 0 && temp_no_telpon.charAt(0) === '0') {
        temp_no_telpon = temp_no_telpon.slice(1)
        temp_no_telpon = `+62 ${temp_no_telpon.substring(0, 3)} ${temp_no_telpon.substring(
          3,
          7
        )} ${temp_no_telpon.substring(7)}`
      }

      setDataKelengkapan({
        nama_pelanggan: store.dataResponseVerifikasiId.data.CUST_NAME,
        customer_id: store.dataResponseVerifikasiId.data.CUST_NUMBER,
        provinsi: '',
        kota: '',
        is_pk: '',
        is_badan_hukum: '',
        is_npwp: '',
        npwp: '',
        ktp: '',
        ktp_file: '',
        nama_ktp: '',
        alamat_ktp: store.dataResponseVerifikasiId.data.ADDRESS_NPWP_NIK,
        alamat_lokasi_pemasangan: '',
        no_telpon: temp_no_telpon,
        email: store.dataResponseVerifikasiId.data.EMAIL,
        media_informasi: {
          sms: false,
          whatsapp: false,
          email: false,
          pgn_mobile: false
        }
      })
    }

    window.scrollTo({
      top: 0,
      behavior: 'smooth'
    })
  }, [])

  const setStateLocal = (key, value, subKey = null) => {
    if (subKey) {
      setDataKelengkapan(prevState => ({
        ...prevState,
        [key]: {
          ...prevState[key],
          [subKey]: value
        }
      }))
    } else {
      setDataKelengkapan(prevState => ({
        ...prevState,
        [key]: value
      }))
    }
  }

  useEffect(() => {
    dispatch(setKelengkapan(dataKelengkapan))
  }, [dataKelengkapan])

  // ASYNC SELECT
  const [openProvinsi, setOpenProvinsi] = useState(false)
  const [optionsProvinsi, setOptionsProvinsi] = useState([])
  const [loadingProvinsi, setLoadingProvinsi] = useState(false)

  const [openKota, setOpenKota] = useState(false)
  const [optionsKota, setOptionsKota] = useState([])
  const [loadingKota, setLoadingKota] = useState(false)

  useEffect(() => {
    let active = true
    if (!loadingProvinsi) {
      return undefined
    }

    setDefaultErrorState()

    if (optionsProvinsi.length == 0) {
      fetchDataProvinsi(active)
    } else {
      setLoadingProvinsi(false)
    }

    return () => {
      active = false
    }
  }, [loadingProvinsi])

  useEffect(() => {
    let active = true
    if (!loadingKota) {
      return undefined
    }

    setDefaultErrorState()

    if (optionsKota.length == 0) {
      fetchDataKota(active)
    } else {
      setLoadingKota(false)
    }

    return () => {
      active = false
    }
  }, [loadingKota])

  const fetchDataProvinsi = async active => {
    const responseProvinsi = await callGet(Endpoint.getListProvince)

    if (!responseProvinsi.status) {
      dispatch(
        setErrorApi({
          data: responseProvinsi.data,
          message: responseProvinsi.message,
          status: true
        })
      )
    } else {
      if (active) {
        setOptionsProvinsi(responseProvinsi.data.data)
      }
    }

    setLoadingProvinsi(false)
  }

  const fetchDataKota = async active => {
    if (dataKelengkapan.provinsi != null) {
      let params = {
        geo_province_id: dataKelengkapan.provinsi.id
      }

      const responseKota = await callGet(Endpoint.getListTown, params)

      if (!responseKota.status) {
        dispatch(
          setErrorApi({
            data: responseKota.data,
            message: responseKota.message,
            status: true
          })
        )
      } else {
        if (active) {
          setOptionsKota(responseKota.data.data)
        }
      }
    }
    setLoadingKota(false)
  }

  const handleSelectProvinsi = (event, newValue) => {
    setOptionsKota([])

    setStateLocal('provinsi', newValue)
  }

  const handleSelectKota = (event, newValue) => {
    setStateLocal('kota', newValue)
  }

  const handleFileChange = (e, action) => {
    const selectedFile = e.target.files[0]
    if (selectedFile) {
      const allowedTypes = ['image/jpeg', 'image/jpg', 'image/png']
      if (!allowedTypes.includes(selectedFile.type)) {
        toast.error('Tipe file yang diperbolehkan hanya berupa Gambar (jpg, jpeg, png).')

        return false
      }

      const maxSizeInBytes = 5 * 1024 * 1024 // 5 MB
      if (selectedFile.size > maxSizeInBytes) {
        toast.error('Ukuran file yang diperbolehkan maksimum 5 MB')

        return false
      }

      setStateLocal(action, selectedFile)
    } else {
      setStateLocal(action, null)
    }
  }

  const styleText = {
    color: 'white'
  }

  return (
    <Fragment>
      <div style={{ padding: '10px', marginLeft: '10px', minHeight: '500px', width: '100%', paddingBottom: '80px' }}>
        <div style={{ textAlign: 'center' }}>
          <h1>Kelengkapan</h1>
          <Typography style={styleText}></Typography>
        </div>
        <Grid container spacing={5}>
          {/* SECTION 1 */}

          <Grid item xs={12} sm={6}>
            <CustomTextField
              fullWidth
              required
              label='Nama Pelanggan yang terdaftar di PGN'
              placeholder='Nama Pelanggan'
              value={dataKelengkapan.nama_pelanggan}
              readOnly={true}
              sx={{ pointerEvents: 'none', '& label': { pointerEvents: 'none' } }}
            />
          </Grid>

          <Grid item xs={12} sm={6}>
            <CustomTextField
              fullWidth
              required
              label='Nomor ID Pelanggan PGN'
              placeholder='123456789'
              value={dataKelengkapan.customer_id}
              readOnly={true}
              sx={{ pointerEvents: 'none', '& label': { pointerEvents: 'none' } }}
            />
          </Grid>

          <Grid item xs={12} sm={6}>
            <CustomAutocomplete
              value={dataKelengkapan.provinsi}
              open={openProvinsi}
              options={optionsProvinsi}
              loading={loadingProvinsi}
              onOpen={() => {
                setOpenProvinsi(true)
                setLoadingProvinsi(true)
              }}
              onClose={() => {
                setOpenProvinsi(false)
                setLoadingProvinsi(false)
              }}
              onChange={handleSelectProvinsi}
              id='select-provinsi'
              getOptionLabel={option => option.name || ''}
              isOptionEqualToValue={(option, value) => option.name === value.name}
              renderInput={params => (
                <CustomTextField
                  {...params}
                  label='Provinsi *'
                  InputProps={{
                    ...params.InputProps,
                    endAdornment: (
                      <Fragment>
                        {loadingProvinsi ? <CircularProgress size={20} /> : null}
                        {params.InputProps.endAdornment}
                      </Fragment>
                    )
                  }}
                  placeholder='Pilih Provinsi'
                />
              )}
            />
          </Grid>

          <Grid item xs={12} sm={6}>
            <CustomAutocomplete
              value={dataKelengkapan.kota}
              open={openKota}
              options={optionsKota}
              loading={loadingKota}
              onOpen={() => {
                setOpenKota(true)
                setLoadingKota(true)
              }}
              onClose={() => {
                setOpenKota(false)
                setLoadingKota(false)
              }}
              onChange={handleSelectKota}
              id='select-kota'
              getOptionLabel={option => option.name || ''}
              isOptionEqualToValue={(option, value) => option.name === value.name}
              renderInput={params => (
                <CustomTextField
                  {...params}
                  label='Kota/Kabupaten *'
                  InputProps={{
                    ...params.InputProps,
                    endAdornment: (
                      <Fragment>
                        {loadingKota ? <CircularProgress size={20} /> : null}
                        {params.InputProps.endAdornment}
                      </Fragment>
                    )
                  }}
                  placeholder='Pilih Kota/Kabupaten'
                />
              )}
            />
            <span style={{ fontSize: '10pt' }}>
              diisi dengan lokasi Kota/Kabupaten tempat Anda berlangganan Gas PGN.
            </span>
          </Grid>

          <Grid item xs={12} sm={6}>
            <div>Apakah Anda Pelanggan Kecil (PK) ? *</div>
            <div>
              <RadioGroup
                row
                value={dataKelengkapan.is_pk}
                name='is-pk-radio'
                onChange={e => setStateLocal('is_pk', e.target.value)}
                aria-label='is-pk-radio'
                sx={{ '& svg': { height: 25, width: 25, stroke: 'white' } }}
              >
                <FormControlLabel
                  value='1'
                  control={<Radio color='success' />}
                  label={<Typography style={{ color: 'white', fontSize: '10pt' }}>Ya</Typography>}
                />
                <FormControlLabel
                  value='0'
                  control={<Radio color='success' />}
                  label={<Typography style={{ color: 'white', fontSize: '10pt' }}>Tidak</Typography>}
                />
              </RadioGroup>
            </div>
          </Grid>

          <Grid item xs={12} sm={6}>
            <span style={{ fontSize: '10pt' }}>
              Pelanggan Kecil (PK) adalah Pelanggan yang menggunakan Gas untuk kegiatan yang bersifat jasa atau
              komersial. Jika Anda menggunakan Gas untuk kebutuhan pribadi atau Rumah Tangga, pilih TIDAK
            </span>
          </Grid>

          {dataKelengkapan.is_pk == 1 && (
            <Grid item xs={12} sm={12}>
              <div>Apakah Usaha Anda Berbadan Hukum ? *</div>
              <div>
                <RadioGroup
                  row
                  value={dataKelengkapan.is_badan_hukum}
                  name='is-badan-hukum-radio'
                  onChange={e => setStateLocal('is_badan_hukum', e.target.value)}
                  aria-label='is-badan-hukum-radio'
                  sx={{ '& svg': { height: 25, width: 25, stroke: 'white' } }}
                >
                  <FormControlLabel
                    value='1'
                    control={<Radio color='success' />}
                    label={<Typography style={{ color: 'white', fontSize: '10pt' }}>Ya</Typography>}
                  />
                  <FormControlLabel
                    value='0'
                    control={<Radio color='success' />}
                    label={<Typography style={{ color: 'white', fontSize: '10pt' }}>Tidak</Typography>}
                  />
                </RadioGroup>
              </div>
            </Grid>
          )}

          <Grid item xs={12} sm={12}>
            <div>Apakah Anda Memiliki NPWP ? *</div>
            <div>
              <RadioGroup
                row
                value={dataKelengkapan.is_npwp}
                name='is-npwp-radio'
                onChange={e => setStateLocal('is_npwp', e.target.value)}
                aria-label='is-npwp-radio'
                sx={{ '& svg': { height: 25, width: 25, stroke: 'white' } }}
              >
                <FormControlLabel
                  value='1'
                  control={<Radio color='success' />}
                  label={<Typography style={{ color: 'white', fontSize: '10pt' }}>Ya</Typography>}
                />
                <FormControlLabel
                  value='0'
                  control={<Radio color='success' />}
                  label={<Typography style={{ color: 'white', fontSize: '10pt' }}>Tidak</Typography>}
                />
              </RadioGroup>
            </div>
          </Grid>

          {dataKelengkapan.is_npwp == 1 && (
            <>
              <Grid item xs={12} sm={6}>
                <CleaveWrapper>
                  <InputLabel style={{ color: 'white' }}>NPWP Pelanggan *</InputLabel>
                  <Cleave
                    options={{ blocks: [16], numericOnly: true }}
                    style={{ backgroundColor: 'white' }}
                    placeholder='0000 0000 0000 0000'
                    value={dataKelengkapan.npwp}
                    onChange={e => setStateLocal('npwp', e.target.value)}
                  />
                </CleaveWrapper>
              </Grid>
              <Grid item xs={12} sm={6} style={{ display: 'flex', alignItems: 'center' }}>
                <span style={{ fontSize: '10pt' }}>Jika Anda Usaha Berbadan Hukum, mohon masukkan NPWP Usaha Anda</span>
              </Grid>
            </>
          )}

          <Grid item xs={12} sm={6}>
            <CleaveWrapper>
              <InputLabel style={{ color: 'white' }}>NIK Pelanggan *</InputLabel>
              <Cleave
                options={{ blocks: [16], numericOnly: true }}
                style={{ backgroundColor: 'white' }}
                placeholder='0000 0000 0000 0000'
                value={dataKelengkapan.ktp}
                onChange={e => setStateLocal('ktp', e.target.value)}
              />
            </CleaveWrapper>
          </Grid>

          <Grid item xs={12} sm={6} style={{ display: 'flex', alignItems: 'center' }}>
            <span style={{ fontSize: '10pt' }}>NIK adalah Nomor Induk Kependudukan atau Nomor KTP</span>
          </Grid>

          <Grid item xs={12} sm={6}>
            <InputLabel htmlFor='file_ktp' style={{ color: 'white' }}>
              Unggah KTP *
            </InputLabel>
            <Button
              id='file_ktp'
              component='label'
              variant='outlined'
              color='primary'
              onChange={e => handleFileChange(e, 'ktp_file')}
              style={{ width: '100%', display: 'flex', justifyContent: 'space-between', borderColor: 'white' }}
            >
              <span id='nama_file' style={{ color: 'white', fontSize: '9pt', textAlign: 'left' }}>
                {dataKelengkapan.ktp_file && dataKelengkapan.ktp_file.name
                  ? dataKelengkapan.ktp_file.name
                  : 'Belum Ada Unggahan'}
              </span>
              <VisuallyHiddenInput type='file' accept='image/jpeg, image/jpg, image/png' />
              <span id='btn_file' style={{ color: 'white', fontSize: '10pt', textAlign: 'right', width: '60px' }}>
                Pilih File
              </span>
            </Button>
          </Grid>

          <Grid item xs={12} sm={6} style={{ display: 'flex', alignItems: 'center' }}>
            <span style={{ fontSize: '10pt' }}>(JPG/PNG Format, Max Size : 4MB)</span>
          </Grid>

          <Grid item xs={12} sm={6}>
            <CustomTextField
              fullWidth
              required
              label='Nama Sesuai NIK / NPWP'
              placeholder='Nama Sesuai NIK / NPWP'
              value={dataKelengkapan.nama_ktp}
              onChange={e => setStateLocal('nama_ktp', e.target.value)}
              readOnly
            />
          </Grid>

          <Grid item xs={12} sm={12}>
            <CustomTextField
              fullWidth
              required
              rows={3}
              multiline
              label='Alamat Sesuai NIK / NPWP'
              placeholder='Alamat Sesuai NIK / NPWP'
              value={dataKelengkapan.alamat_ktp}
              onChange={e => setStateLocal('alamat_ktp', e.target.value)}
            />
          </Grid>

          <Grid item xs={12} sm={12}>
            <CustomTextField
              fullWidth
              required
              rows={3}
              multiline
              label='Alamat Lokasi Pemasangan'
              placeholder='Alamat Lokasi Pemasangan'
              value={dataKelengkapan.alamat_lokasi_pemasangan}
              onChange={e => setStateLocal('alamat_lokasi_pemasangan', e.target.value)}
            />
          </Grid>

          <Grid item xs={12} sm={6}>
            <CleaveWrapper>
              <InputLabel htmlFor='no-telp' style={{ color: 'white' }}>
                Nomor HP/WhatsApp *
              </InputLabel>
              <Cleave
                id='no-telp'
                options={{ prefix: '+62', blocks: [3, 3, 4, 4], numericOnly: true }}
                style={{ backgroundColor: 'white' }}
                placeholder='000 0000 0000'
                value={dataKelengkapan.no_telpon}
                onChange={e => setStateLocal('no_telpon', e.target.value)}
              />
            </CleaveWrapper>
          </Grid>

          <Grid item xs={12} sm={6} style={{ display: 'flex', alignItems: 'center' }}>
            <span style={{ fontSize: '10pt' }}>
              Silakan masukkan Nomor Handphone atau WhatsApp Anda (Contoh: 628123456789)
            </span>
          </Grid>

          <Grid item xs={12} sm={6}>
            <CustomTextField
              id='email'
              fullWidth
              label='Email'
              placeholder='example@mail.com'
              value={dataKelengkapan.email}
              onChange={e => setStateLocal('email', e.target.value)}
            />
          </Grid>

          <Grid item xs={12} sm={6} style={{ display: 'flex', alignItems: 'center' }}>
            <span style={{ fontSize: '10pt' }}>Silakan masukkan Email aktif Anda</span>
          </Grid>

          <Grid item xs={12} sm={6}>
            <div>Media Informasi Layanan dari PGN yang dipilih, Misal : Informasi Tagihan Gas</div>
          </Grid>

          <Grid item xs={12} sm={6}>
            <div>
              <FormControlLabel
                id='media-informasi-sms'
                sx={{
                  '& svg': { height: 25, width: 25, stroke: 'white' },
                  '& .MuiFormControlLabel-label': { fontSize: '11pt' }
                }}
                label='SMS (dikenakan biaya)'
                control={
                  <Checkbox
                    checked={dataKelengkapan.media_informasi.sms}
                    onChange={e => setStateLocal('media_informasi', e.target.checked, 'sms')}
                  />
                }
              />
            </div>
            <div>
              <FormControlLabel
                id='media-informasi-whatsapp'
                className='media_informasi'
                sx={{
                  '& svg': { height: 25, width: 25, stroke: 'white' },
                  '& .MuiFormControlLabel-label': { fontSize: '11pt' }
                }}
                label='Whatsapp'
                control={
                  <Checkbox
                    checked={dataKelengkapan.media_informasi.whatsapp}
                    onChange={e => setStateLocal('media_informasi', e.target.checked, 'whatsapp')}
                  />
                }
              />
            </div>
            <div>
              <FormControlLabel
                id='media-informasi-email'
                className='media_informasi'
                sx={{
                  '& svg': { height: 25, width: 25, stroke: 'white' },
                  '& .MuiFormControlLabel-label': { fontSize: '11pt' }
                }}
                label='e-Mail'
                control={
                  <Checkbox
                    checked={dataKelengkapan.media_informasi.email}
                    onChange={e => setStateLocal('media_informasi', e.target.checked, 'email')}
                  />
                }
              />
            </div>
            <div>
              <FormControlLabel
                id='media-informasi-pgn-mobile'
                className='media_informasi'
                sx={{
                  '& svg': { height: 25, width: 25, stroke: 'white' },
                  '& .MuiFormControlLabel-label': { fontSize: '11pt' }
                }}
                label='PGN Mobile (download melalui Apps Store dan Play Store)'
                control={
                  <Checkbox
                    checked={dataKelengkapan.media_informasi.pgn_mobile}
                    onChange={e => setStateLocal('media_informasi', e.target.checked, 'pgn_mobile')}
                  />
                }
              />
            </div>
          </Grid>
        </Grid>
      </div>
    </Fragment>
  )
}

export default FormKelengkapan
